////////////////////////////////////////////////////////////////////////////////
// Simons' Basic.cpp -- this file is part of the Emulator Developers Kit
// available at http://ourworld.compuserve.com/homepages/pc64/develop.htm
//
// The Simons' Basic cartridge contains:
// - 8 kb ROM at $8000
// - 8 kb ROM at $A000
// - a banking register at $DExx to switch ROMs at $A000
//   read for Basic V2, write any value for Simons' Basic
//
// Note that the name of the author is David Simons, not Simon ;-)
// Simons' Basic ROMs Copyright (C)1983 D.S. Software
//
// In Visual C++ "Tools / Options / Directories", add
//   Include files                = C:\PC64Win\EDK\Include
//
// In Visual C++ "Build / Settings / Debug", set
//   Executable for debug session = ..\PC64Win.exe
//   Working Directory            = ..
//   Program arguments            = -extension SimonBas.dll

#include <EDK.h>


// OnRead/OnWrite functions require a class
class SimonsBasic : public Object {

  static const byte abROM8000[8192];
  static const byte abROMA000[8192];

  C64* pC64;
  CPU6510* pCPU;
  byte* pbIO;
  byte* pb8000;
  byte* pbA000;
  byte bTrapDExx;

public:

  // switch ROMs at A000
  void WriteDExx(byte) {
    pC64->MapROMA000(pbA000);
  }
  byte ReadDExx() {
    pC64->UnmapROMA000();
    return 0;
  }

  // destructor
  void CleanUp() {
    if (bTrapDExx != 0) {
      for (int i = 0; i < 256; i++) {
        pCPU->SetTrap(pbIO + 0x0E00 + i, 0);
      }
      pCPU->FreeTrap(bTrapDExx);
      bTrapDExx = 0;
    }
    if (pb8000 != NULL) {
      pC64->UnmapROM8000();
      pCPU->FreeMem(pb8000, 8192);
      pb8000 = NULL;
    }
    if (pbA000 != NULL) {
      pC64->UnmapROMA000();
      pCPU->FreeMem(pbA000, 8192);
      pbA000 = NULL;
    }
  }
  virtual ~SimonsBasic() {
    CleanUp();
  }

  // constructor
  SimonsBasic() {
    pC64 = NULL;
    pCPU = NULL;
    pbIO = NULL;
    pb8000 = NULL;
    pbA000 = NULL;
    bTrapDExx = 0;
    try {
      pC64 = (C64*)gObjectRoot.FindChild("C64");
      pCPU = (CPU6510*)pC64->FindChild("CPU");
      pbIO = pC64->GetIO();
      pb8000 = pCPU->AllocMem(8192);
      memcpy(pb8000, abROM8000, 8192);
      pC64->MapROM8000(pb8000);
      pbA000 = pCPU->AllocMem(8192);
      memcpy(pbA000, abROMA000, 8192);
      bTrapDExx = pCPU->AllocIOTrapWithDummyReads(this, (bpfn)ReadDExx, (pfnb)WriteDExx);
      for (int i = 0; i < 256; i++) {
        pCPU->SetTrap(pbIO + 0x0E00 + i, bTrapDExx);
      }
    } catch (...) {
      CleanUp();
      throw;
    }
  }

};


// DLL entry point
BOOL WINAPI DllMain(HINSTANCE, DWORD dwReason, LPVOID) {
  try {
    static SimonsBasic* pSimonsBasic;
    switch (dwReason) {
    case DLL_PROCESS_ATTACH:
      assert(pSimonsBasic == NULL);
      pSimonsBasic = new SimonsBasic();
      break;
    case DLL_PROCESS_DETACH:
      if (pSimonsBasic != NULL) {
        delete pSimonsBasic;
        pSimonsBasic = NULL;
      }
      break;
    }
    return TRUE;
  } catch (...) {
    report();
    return FALSE;
  }
}


////////////////////////////////////////////////////////////////////////////////
// Simons' Basic ROMs Copyright (C)1983 D.S. Software

const byte SimonsBasic::abROM8000[8192] = {
  0x52,0x81,0x52,0x81,0xC3,0xC2,0xCD,0x38,0x30,0x41,0x4C,0x52,0x81,0x20,0x2C,0x81,
  0x20,0x8A,0xAD,0x4C,0x3C,0x81,0x20,0x2C,0x81,0x20,0xF7,0xB7,0x4C,0x3C,0x81,0x20,
  0x2C,0x81,0x20,0x6B,0xE2,0x4C,0x3C,0x81,0x20,0x2C,0x81,0x20,0x64,0xE2,0x4C,0x3C,
  0x81,0x20,0x2C,0x81,0x20,0xFD,0xAE,0x4C,0x3C,0x81,0x20,0x2C,0x81,0x20,0x9E,0xAD,
  0x4C,0x3C,0x81,0x20,0x2C,0x81,0x20,0xA3,0xB6,0x4C,0x3C,0x81,0x20,0x2C,0x81,0x20,
  0xD1,0xBD,0x4C,0x3C,0x81,0x20,0x2C,0x81,0x20,0xA3,0xA8,0x4C,0x3C,0x81,0x20,0x2C,
  0x81,0x20,0xFF,0xAE,0x4C,0x3C,0x81,0x20,0x2C,0x81,0x20,0xFD,0xAE,0x4C,0x3C,0x81,
  0x20,0x2C,0x81,0x20,0x13,0xA6,0x4C,0x3C,0x81,0x4C,0xF3,0xBC,0x4C,0x92,0xAE,0x20,
  0x2C,0x81,0x20,0xF7,0xAE,0x4C,0x3C,0x81,0x20,0x2C,0x81,0x20,0x49,0xBC,0x4C,0x3C,
  0x81,0x20,0x2C,0x81,0x20,0x0C,0xBC,0x4C,0x3C,0x81,0x20,0x2C,0x81,0x20,0x30,0xBA,
  0x4C,0x3C,0x81,0x20,0x2C,0x81,0x20,0xA2,0xBB,0x4C,0x3C,0x81,0x20,0x2C,0x81,0x20,
  0x14,0xBB,0x4C,0x3C,0x81,0x20,0x2C,0x81,0x20,0xDF,0xBD,0x4C,0x3C,0x81,0x20,0x2C,
  0x81,0x20,0x28,0xAF,0x4C,0x3C,0x81,0x20,0x2C,0x81,0x20,0xA6,0xB3,0x4C,0x3C,0x81,
  0x20,0x2C,0x81,0x20,0xB5,0xAB,0x4C,0x3C,0x81,0x20,0x2C,0x81,0x20,0xF8,0xA8,0x4C,
  0x3C,0x81,0x20,0x2C,0x81,0x20,0x06,0xA9,0x4C,0x3C,0x81,0x20,0x2C,0x81,0x20,0x0D,
  0xAC,0x4C,0x3C,0x81,0x20,0x2C,0x81,0x20,0xCA,0xAA,0x4C,0x3C,0x81,0x20,0x2C,0x81,
  0x20,0xFB,0xA8,0x4C,0x3C,0x81,0x20,0x2C,0x81,0x20,0x9E,0xB7,0x4C,0x3C,0x81,0x20,
  0x2C,0x81,0x20,0xA3,0xA8,0x4C,0x3C,0x81,0xA5,0x01,0x8D,0x8D,0xC5,0x20,0xD4,0x88,
  0x4C,0x43,0xFE,0x20,0x2C,0x81,0x20,0xBE,0xE1,0x4C,0x3C,0x81,0x8D,0x7F,0xC5,0xAD,
  0xC5,0xCB,0x8D,0x98,0xC5,0x20,0x27,0x83,0xAD,0x7F,0xC5,0x60,0x8D,0x7F,0xC5,0xAD,
  0x98,0xC5,0xD0,0x07,0x20,0x27,0x83,0xAD,0x7F,0xC5,0x60,0x20,0x1E,0x83,0xAD,0x7F,
  0xC5,0x60,0x20,0x27,0x83,0x78,0x20,0x84,0xFF,0x20,0x87,0xFF,0x20,0x8A,0xFF,0x20,
  0x81,0xFF,0x58,0x20,0x53,0xE4,0x20,0xBF,0xE3,0xA0,0x00,0x84,0x22,0xA2,0x80,0x86,
  0x23,0xB1,0x22,0x91,0x22,0xC8,0xD0,0xF9,0xE8,0xE0,0xC0,0xD0,0xF2,0xA9,0x0F,0x8D,
  0x21,0xD0,0xA9,0x06,0x8D,0x20,0xD0,0x20,0x1E,0x83,0xA2,0x00,0xA9,0x00,0x9D,0x00,
  0xC3,0x9D,0x00,0xC4,0x9D,0x00,0xC5,0x9D,0x00,0xC6,0x9D,0x00,0xC7,0x9D,0x00,0xC8,
  0x9D,0x00,0xC9,0x9D,0x00,0xCA,0x9D,0x00,0xCB,0xBD,0x95,0xBF,0x9D,0x92,0x92,0xE8,
  0xD0,0xDA,0xA9,0x09,0x20,0x27,0x83,0xA9,0x86,0x8D,0x09,0x03,0xA9,0x46,0x8D,0x0A,
  0x03,0xA9,0x96,0x8D,0x0B,0x03,0xBD,0x35,0x82,0xF0,0x06,0x20,0xD2,0xFF,0xE8,0xD0,
  0xF5,0xA9,0x80,0xEA,0x20,0x1E,0x83,0x20,0x76,0xBF,0x20,0x27,0x83,0xA9,0x82,0x8D,
  0x05,0x03,0xA9,0x61,0x8D,0x04,0x03,0xA9,0x8F,0x8D,0x01,0x03,0xA9,0xAE,0x8D,0x00,
  0x03,0xA9,0x82,0x8D,0x03,0x03,0xA9,0x89,0x8D,0x02,0x03,0xA9,0x83,0x8D,0x19,0x03,
  0xA9,0x30,0x8D,0x18,0x03,0xA9,0x83,0x8D,0x17,0x03,0xA9,0x53,0x8D,0x16,0x03,0xA5,
  0x2B,0xA4,0x2C,0x20,0x08,0xA4,0x20,0x30,0xE4,0xA9,0x18,0x8D,0xFA,0xFF,0xA9,0x81,
  0x8D,0xFB,0xFF,0xA2,0xFB,0x9A,0x4C,0x86,0xE3,0x20,0x31,0x80,0x4C,0x06,0x81,0x20,
  0x4E,0x90,0x4C,0x06,0x81,0x90,0x93,0x2A,0x2A,0x2A,0x20,0x45,0x58,0x54,0x45,0x4E,
  0x44,0x45,0x44,0x20,0x43,0x42,0x4D,0x20,0x56,0x32,0x20,0x42,0x41,0x53,0x49,0x43,
  0x20,0x2A,0x2A,0x2A,0x0D,0x0D,0x00,0x20,0xD4,0x88,0x58,0x20,0x27,0x83,0x4C,0xAE,
  0xA7,0x20,0x1E,0x83,0x20,0x2B,0xAD,0x20,0x27,0x83,0x4C,0x7C,0xA5,0x08,0x48,0x20,
  0x1E,0x83,0x68,0x28,0x4C,0x16,0xBE,0x48,0x20,0x27,0x83,0x68,0x28,0x4C,0x1A,0xA7,
  0x48,0x20,0x27,0x83,0x68,0x28,0x4C,0xF6,0xA6,0x78,0xA9,0x9C,0x8D,0x15,0x03,0xA9,
  0xDC,0x8D,0x14,0x03,0x8D,0x8F,0xCB,0xA9,0x80,0x8D,0x91,0x02,0xA9,0x00,0x8D,0x1A,
  0xD0,0x58,0xAD,0x18,0xD0,0xC9,0x0B,0xD0,0x06,0x20,0x1E,0x83,0x20,0xF9,0xA9,0x20,
  0x27,0x83,0xAD,0xB0,0xC5,0xC9,0x0A,0xF0,0x03,0x4C,0x83,0xA4,0x18,0xAD,0xFE,0x01,
  0x6D,0xAF,0xC5,0x85,0x63,0x8D,0xFE,0x01,0xAD,0xFF,0x01,0x69,0x00,0x85,0x62,0x8D,
  0xFF,0x01,0xA2,0x90,0x38,0x20,0x49,0xBC,0x20,0xDF,0xBD,0xA2,0x00,0xBD,0x00,0x01,
  0xF0,0x06,0x9D,0x77,0x02,0xE8,0xD0,0xF5,0xA9,0x20,0x9D,0x77,0x02,0xE8,0x86,0xC6,
  0xA9,0x00,0x8D,0xB0,0xC5,0x20,0x60,0xA5,0x84,0xAA,0xA0,0x00,0xC8,0xB9,0x00,0x02,
  0xF0,0x12,0xC9,0x3A,0x90,0xF6,0xC9,0x5B,0xB0,0xF2,0xA4,0xAA,0xA9,0x0A,0x8D,0xB0,
  0xC5,0x4C,0x86,0xA4,0xA9,0x00,0x8D,0xB0,0xC5,0xA4,0xAA,0x4C,0x86,0xA4,0x8D,0x00,
  0xDE,0xA9,0x0A,0x8D,0xC5,0xCB,0x60,0xAD,0x00,0xDE,0xA9,0x00,0x8D,0xC5,0xCB,0x60,
  0x48,0x8A,0x48,0x98,0x48,0x20,0xD4,0x88,0xAD,0xC5,0xCB,0x8D,0x01,0xC5,0x20,0x27,
  0x83,0xA9,0x7F,0x8D,0x0D,0xDD,0xAC,0x0D,0xDD,0x30,0x32,0x20,0xBC,0xF6,0x20,0xE1,
  0xFF,0xD0,0x2A,0xA9,0x04,0x8D,0x88,0x02,0x20,0xA3,0xFD,0x20,0x18,0xE5,0xA9,0x0F,
  0x8D,0x21,0xD0,0xA9,0x06,0x8D,0x20,0xD0,0xA9,0x00,0x8D,0x86,0x02,0x85,0xD4,0x8D,
  0xB2,0xC5,0x8D,0xB1,0xC5,0xA9,0x08,0x8D,0x8F,0xCB,0x6C,0x02,0xA0,0xAD,0x01,0xC5,
  0xD0,0x06,0x20,0x27,0x83,0x4C,0x72,0xFE,0x20,0x1E,0x83,0x4C,0x72,0xFE,0x20,0x4E,
  0x90,0x20,0x27,0x83,0xA9,0x00,0x85,0x0A,0x20,0xD4,0xE1,0x38,0xA5,0x2D,0xE9,0x02,
  0xAA,0xA5,0x2E,0xE9,0x00,0xA8,0xA9,0x00,0x20,0xD5,0xFF,0xB0,0x19,0x20,0xB7,0xFF,
  0x29,0xBF,0xF0,0x05,0xA2,0x1D,0x4C,0x37,0xA4,0x86,0x2D,0x84,0x2E,0x20,0x59,0xA6,
  0x20,0x33,0xA5,0x4C,0x74,0xA4,0x4C,0xF9,0xE0,0xA9,0x01,0x85,0xA8,0xA9,0x08,0x85,
  0xA9,0x20,0x4E,0x90,0x20,0xF0,0x88,0x84,0xA4,0x85,0xA5,0x20,0x31,0x80,0x20,0xF0,
  0x88,0x8C,0xAA,0xC5,0x8D,0xAB,0xC5,0xA0,0x00,0xB1,0xA8,0x85,0x09,0x20,0x61,0x89,
  0xB1,0xA8,0x85,0x0A,0x20,0x61,0x89,0xA5,0xA4,0x91,0xA8,0x20,0x61,0x89,0xA5,0xA5,
  0x91,0xA8,0xA5,0x09,0x05,0x0A,0xF0,0x1A,0xA5,0x09,0x85,0xA8,0xA5,0x0A,0x85,0xA9,
  0x18,0xA5,0xA4,0x6D,0xAA,0xC5,0x85,0xA4,0xA5,0xA5,0x6D,0xAB,0xC5,0x85,0xA5,0x4C,
  0xE9,0x83,0x4C,0x57,0x82,0xAD,0xA8,0xC5,0xC9,0x9F,0x90,0x05,0xA9,0x9F,0x8D,0xA8,
  0xC5,0xAD,0xA6,0xC5,0xC9,0xC7,0x90,0x05,0xA9,0xC7,0x8D,0xA6,0xC5,0xA5,0x09,0x85,
  0x6F,0x29,0x03,0x0A,0xAA,0x4C,0x5B,0x84,0xA5,0x09,0x29,0x03,0x0A,0xAA,0xD0,0x0B,
  0x18,0xA5,0xA8,0x69,0x08,0x85,0xA8,0x90,0x02,0xE6,0xA9,0xA5,0xA4,0x29,0x07,0xA8,
  0x20,0xCC,0x88,0xEA,0x20,0x92,0x92,0xD0,0x0C,0xB1,0xA8,0x1D,0x4A,0x88,0xE8,0x3D,
  0x60,0x88,0x4C,0xAC,0x84,0xC9,0x00,0xD0,0x0C,0xB1,0xA8,0x3D,0x60,0x88,0xE8,0x3D,
  0x60,0x88,0x4C,0xAC,0x84,0xC9,0x01,0xD0,0x0B,0xB1,0xA8,0x3D,0x60,0x88,0xE8,0x1D,
  0x4A,0x88,0xD0,0x18,0xC9,0x04,0xF0,0x0B,0xB1,0xA8,0x1D,0x4A,0x88,0xE8,0x1D,0x4A,
  0x88,0xD0,0x09,0xB1,0xA8,0x5D,0x4A,0x88,0xE8,0x5D,0x4A,0x88,0x91,0xA8,0xE6,0x09,
  0xA5,0x09,0xC9,0xA0,0xB0,0x07,0xCD,0xA8,0xC5,0x90,0x8D,0xF0,0x8B,0xE6,0xA4,0xA5,
  0xA4,0x29,0x07,0xD0,0x0D,0x18,0xA5,0xAA,0x69,0x40,0x85,0xAA,0xA5,0xAB,0x69,0x01,
  0x85,0xAB,0xA5,0xAA,0x85,0xA8,0xA5,0xAB,0x85,0xA9,0xA5,0x6F,0x85,0x09,0xA5,0xA4,
  0xCD,0xA6,0xC5,0xF0,0x02,0xB0,0x03,0x4C,0x3D,0x84,0x4C,0xD4,0x88,0x20,0x7B,0x88,
  0x20,0xF6,0x84,0x4C,0x57,0x82,0xAE,0xAA,0xC5,0xAD,0x99,0xC5,0x8D,0xAA,0xC5,0x8E,
  0x99,0xC5,0xAE,0xAB,0xC5,0xAD,0x9A,0xC5,0x8D,0xAB,0xC5,0x8E,0x9A,0xC5,0xAD,0xAB,
  0xC5,0x85,0x0A,0xAD,0x9A,0xC5,0x85,0xA5,0xAD,0xAA,0xC5,0x85,0x09,0xAD,0x99,0xC5,
  0x85,0xA4,0xA5,0xF7,0x48,0xA9,0x0B,0x85,0xF7,0x85,0x90,0x20,0x32,0x87,0x68,0x85,
  0xF7,0xA5,0x90,0xC9,0x08,0xD0,0x05,0xA9,0x00,0x4C,0x4A,0x92,0xAD,0xAA,0xC5,0x85,
  0x09,0xAD,0xAB,0xC5,0x85,0x0A,0xAD,0x99,0xC5,0x85,0xA4,0xA5,0xA8,0x85,0xAA,0xA5,
  0xA9,0x85,0xAB,0x20,0xCC,0x88,0xAD,0xAE,0xC5,0xC9,0x2C,0xD0,0x03,0x4C,0x25,0x84,
  0xA5,0x09,0x85,0x6F,0xA5,0x0A,0x85,0x70,0xA5,0x09,0x29,0x07,0xAA,0x4C,0x82,0x85,
  0xA5,0x09,0x29,0x07,0xAA,0xD0,0x0B,0x18,0xA5,0xA8,0x69,0x08,0x85,0xA8,0x90,0x02,
  0xE6,0xA9,0xA5,0xA4,0x29,0x07,0xA8,0x20,0xCC,0x88,0xEA,0x20,0x92,0x92,0xD0,0x0A,
  0xB1,0xA8,0x5D,0x4A,0x88,0x91,0xA8,0x4C,0xAE,0x85,0xC9,0x01,0xD0,0x09,0xB1,0xA8,
  0x1D,0x4A,0x88,0x91,0xA8,0xD0,0x07,0xB1,0xA8,0x3D,0x60,0x88,0x91,0xA8,0xE6,0x09,
  0xD0,0x02,0xE6,0x0A,0xA5,0x0A,0xF0,0x06,0xA5,0x09,0xC9,0x40,0xB0,0x0D,0x18,0xA5,
  0x09,0xED,0xA8,0xC5,0xA5,0x0A,0xED,0xA9,0xC5,0x90,0xA5,0xE6,0xA4,0xA5,0xA4,0x29,
  0x07,0xD0,0x0D,0x18,0xA5,0xAA,0x69,0x40,0x85,0xAA,0xA5,0xAB,0x69,0x01,0x85,0xAB,
  0xA5,0xAA,0x85,0xA8,0xA5,0xAB,0x85,0xA9,0xA5,0x6F,0x85,0x09,0xA5,0x70,0x85,0x0A,
  0xA5,0xA4,0xC9,0xC8,0xB0,0x07,0xCD,0xA6,0xC5,0xF0,0x05,0x90,0x03,0x4C,0xD4,0x88,
  0x4C,0x68,0x85,0x20,0x73,0x00,0x4C,0x57,0x82,0xAD,0xFF,0xC5,0xC9,0x0A,0xD0,0x21,
  0xA5,0x39,0xCD,0x88,0xC5,0xD0,0x07,0xA5,0x3A,0xCD,0x89,0xC5,0xF0,0x1D,0xA5,0x39,
  0x8D,0x88,0xC5,0xA5,0x3A,0x8D,0x89,0xC5,0x20,0x1E,0x83,0x20,0x28,0x93,0x20,0x27,
  0x83,0xAD,0xE8,0xC5,0xC9,0x0A,0xD0,0x03,0x20,0xCC,0x8E,0x20,0x73,0x00,0xA9,0x00,
  0x8D,0x50,0xCA,0x8D,0xB0,0xC5,0xAD,0xFF,0xC5,0xC9,0x0A,0xD0,0x09,0x20,0x1E,0x83,
  0x20,0xA0,0x93,0x20,0x27,0x83,0xA0,0x00,0xB1,0x7A,0xC9,0x8B,0xD0,0x03,0x4C,0x7E,
  0x8C,0xC9,0x64,0xF0,0x20,0xC9,0x8A,0xF0,0x0E,0xC9,0x9C,0xF0,0x0A,0xC9,0xA2,0xF0,
  0x06,0x20,0x79,0x00,0x4C,0xE7,0xA7,0xA2,0x00,0x8E,0x15,0xC6,0x8E,0x2A,0xC6,0x8E,
  0x3F,0xC6,0x4C,0x71,0x86,0x20,0x4E,0x90,0x20,0x1E,0x83,0xB1,0x7A,0x0A,0xA8,0x88,
  0x88,0xB9,0x66,0x90,0x48,0xB9,0x65,0x90,0x48,0x60,0xA9,0x01,0x8D,0x30,0xC5,0xA9,
  0x3F,0x8D,0x2F,0xC5,0xA9,0x00,0x85,0xA6,0xA0,0x00,0x8C,0x1B,0xCB,0xA9,0xE0,0x85,
  0xA7,0x20,0x14,0xA0,0xA6,0xAA,0x86,0xAC,0xA9,0x00,0x91,0xA6,0x20,0x5A,0x89,0xA5,
  0xA6,0xC9,0x40,0xD0,0xF3,0xA5,0xA7,0xC9,0xFF,0xD0,0xED,0xA9,0x00,0x8D,0xAE,0xC5,
  0x20,0xF9,0xA9,0xA9,0x0A,0x8D,0xB1,0xC5,0xA9,0x3B,0x8D,0x11,0xD0,0xA9,0x0B,0x8D,
  0x18,0xD0,0xA9,0x94,0x8D,0x00,0xDD,0xA0,0x00,0x8C,0x1B,0xCB,0xA5,0xAC,0x99,0x00,
  0xC0,0x99,0xFF,0xC0,0x99,0xFE,0xC1,0x99,0xFD,0xC2,0xC8,0xC0,0xFF,0xD0,0xED,0x4C,
  0x57,0x82,0x20,0x4E,0x90,0x20,0xF0,0x88,0x84,0x09,0x85,0x0A,0x20,0x29,0x82,0x86,
  0xA4,0x20,0x9C,0x88,0xA5,0x0A,0xF0,0x0F,0xA5,0x09,0xC9,0x7F,0xD0,0x09,0xA5,0xF7,
  0xC9,0x7F,0xD0,0x03,0x4C,0xF5,0xBC,0x20,0x32,0x87,0x4C,0x57,0x82,0xA9,0x08,0x85,
  0x90,0x60,0x18,0xA5,0x09,0xED,0x2F,0xC5,0xA5,0x0A,0xED,0x30,0xC5,0xB0,0xEE,0xA5,
  0xA4,0xC9,0xC8,0xB0,0xE8,0xAD,0xAE,0xC5,0xF0,0x04,0x06,0x09,0x26,0x0A,0xA5,0x09,
  0x29,0x07,0xAA,0xA5,0xA4,0x29,0x07,0x85,0x63,0xA5,0xA4,0x4A,0x4A,0x4A,0xA8,0xA5,
  0x09,0x29,0xF8,0x85,0xA8,0xA5,0xF7,0xC9,0x0B,0xF0,0x36,0xAD,0x1B,0xCB,0xF0,0x31,
  0x84,0xCD,0xA5,0x0A,0x4A,0x66,0x09,0x4A,0x66,0x09,0x4A,0x66,0x09,0x18,0xB9,0x28,
  0x89,0x65,0x09,0x85,0xAA,0xB9,0x41,0x89,0x69,0xC0,0x85,0xAB,0xA0,0x00,0xAD,0x19,
  0xCB,0x91,0xAA,0x18,0xA5,0xAB,0x69,0x18,0x85,0xAB,0xAD,0x1A,0xCB,0x91,0xAA,0xA4,
  0xCD,0x18,0xB9,0xF6,0x88,0x65,0xA8,0x85,0xA8,0xB9,0x0F,0x89,0x65,0x0A,0x85,0xA9,
  0xA4,0x63,0x78,0xA5,0x01,0x29,0xFD,0x85,0x01,0xAD,0xAE,0xC5,0xF0,0x76,0xA5,0xF7,
  0xC9,0x0B,0xF0,0x54,0xC9,0x00,0xD0,0x0E,0xB1,0xA8,0x3D,0x60,0x88,0xE8,0x3D,0x60,
  0x88,0x91,0xA8,0x4C,0xD4,0x88,0xC9,0x01,0xD0,0x0D,0xB1,0xA8,0x3D,0x60,0x88,0xE8,
  0x1D,0x4A,0x88,0x91,0xA8,0xD0,0x46,0xC9,0x02,0xD0,0x0E,0xB1,0xA8,0x1D,0x4A,0x88,
  0xE8,0x3D,0x60,0x88,0x91,0xA8,0x4C,0xD4,0x88,0xC9,0x03,0xD0,0x0D,0xB1,0xA8,0x1D,
  0x4A,0x88,0xE8,0x1D,0x4A,0x88,0x91,0xA8,0xD0,0x23,0xB1,0xA8,0x5D,0x4A,0x88,0xE8,
  0x5D,0x4A,0x88,0x91,0xA8,0x4C,0xD4,0x88,0xB1,0xA8,0x3D,0x4A,0x88,0xF0,0x04,0xA9,
  0x02,0x85,0x90,0xE8,0xB1,0xA8,0x3D,0x4A,0x88,0xF0,0x02,0xE6,0x90,0xA5,0x01,0x09,
  0x02,0x85,0x01,0x60,0xA5,0xF7,0xF0,0x1A,0xC9,0x01,0xF0,0x07,0xC9,0x02,0xF0,0x28,
  0x4C,0x77,0x8B,0xB1,0xA8,0x1D,0x4A,0x88,0xD0,0x0D,0x80,0x40,0x20,0x10,0x08,0x04,
  0x02,0x01,0xB1,0xA8,0x3D,0x60,0x88,0x91,0xA8,0xA5,0x01,0x09,0x02,0x85,0x01,0x60,
  0x7F,0xBF,0xDF,0xEF,0xF7,0xFB,0xFD,0xFE,0xB1,0xA8,0x5D,0x4A,0x88,0x91,0xA8,0x4C,
  0xD4,0x88,0x20,0x7B,0x88,0x20,0x96,0xA2,0x4C,0x57,0x82,0x20,0x4E,0x90,0x20,0xF0,
  0x88,0x8D,0x9A,0xC5,0x8C,0x99,0xC5,0x20,0x29,0x82,0x8E,0xAA,0xC5,0x20,0xED,0x88,
  0x8D,0xA9,0xC5,0x8C,0xA8,0xC5,0x20,0x29,0x82,0x8E,0xA6,0xC5,0x20,0x29,0x82,0x86,
  0xF7,0x60,0x20,0x2F,0x82,0x8E,0x19,0xCB,0x0E,0x19,0xCB,0x0E,0x19,0xCB,0x0E,0x19,
  0xCB,0x0E,0x19,0xCB,0x20,0x29,0x82,0x8A,0x0D,0x19,0xCB,0x8D,0x19,0xCB,0xA9,0x0A,
  0x8D,0x1B,0xCB,0x20,0x29,0x82,0x8E,0x1A,0xCB,0x4C,0x57,0x82,0x78,0xA5,0x01,0x29,
  0xFD,0x85,0x01,0x60,0xA5,0x01,0x09,0x02,0x85,0x01,0x60,0xAD,0x98,0xC5,0x48,0x20,
  0xF0,0x88,0x8D,0x7F,0xC5,0x68,0x8D,0x98,0xC5,0xAD,0x7F,0xC5,0x60,0x20,0x31,0x80,
  0x20,0x0D,0x80,0x4C,0x16,0x80,0x00,0x40,0x80,0xC0,0x00,0x40,0x80,0xC0,0x00,0x40,
  0x80,0xC0,0x00,0x40,0x80,0xC0,0x00,0x40,0x80,0xC0,0x00,0x40,0x80,0xC0,0x00,0xE0,
  0xE1,0xE2,0xE3,0xE5,0xE6,0xE7,0xE8,0xEA,0xEB,0xEC,0xED,0xEF,0xF0,0xF1,0xF2,0xF4,
  0xF5,0xF6,0xF7,0xF9,0xFA,0xFB,0xFC,0xFE,0x00,0x28,0x50,0x78,0xA0,0xC8,0xF0,0x18,
  0x40,0x68,0x90,0xB8,0xE0,0x08,0x30,0x58,0x80,0xA8,0xD0,0xF8,0x20,0x48,0x70,0x98,
  0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x03,0x03,0x03,0x03,0x03,0xE6,0xA6,0xD0,0x02,0xE6,0xA7,
  0x60,0xE6,0xA8,0xD0,0x02,0xE6,0xA9,0x60,0xA5,0x01,0x09,0x04,0x85,0x01,0x60,0xA5,
  0x01,0x78,0x29,0xFB,0x85,0x01,0x60,0x4C,0x57,0x82,0x20,0x4E,0x90,0x20,0x58,0x90,
  0x86,0xAC,0x85,0xAD,0xA5,0x69,0x48,0x20,0xED,0x88,0x8C,0x6B,0xCB,0x8D,0x6C,0xCB,
  0x20,0x29,0x82,0x8E,0xA3,0xC5,0x20,0x9C,0x88,0x68,0x85,0x5F,0xA9,0x00,0x8D,0x4D,
  0xCB,0xAD,0x04,0xC5,0x8D,0x05,0xC5,0x8C,0x4E,0xCB,0x8E,0x4F,0xCB,0xAC,0x4D,0xCB,
  0xB1,0xAC,0xC4,0x5F,0xF0,0xC1,0xC8,0x18,0xE9,0x2F,0x48,0xCE,0x05,0xC5,0xAD,0x05,
  0xC5,0xD0,0x09,0xAD,0x04,0xC5,0x8D,0x05,0xC5,0x8C,0x4D,0xCB,0xAE,0x4F,0xCB,0x68,
  0xAC,0x4E,0xCB,0xC9,0x09,0xB0,0xA0,0x8D,0x5F,0xCB,0xC9,0x05,0x90,0x03,0x18,0xE9,
  0x04,0xAA,0xBD,0x57,0xCB,0x8D,0x63,0xCB,0xBD,0x5B,0xCB,0x8D,0x67,0xCB,0xAD,0x63,
  0xCB,0xF0,0x1A,0xC9,0xFF,0xD0,0x0E,0xAD,0x6B,0xCB,0xD0,0x03,0xCE,0x6C,0xCB,0xCE,
  0x6B,0xCB,0x4C,0x0D,0x8A,0xEE,0x6B,0xCB,0xD0,0x03,0xEE,0x6C,0xCB,0xAD,0x67,0xCB,
  0xF0,0x0D,0xC9,0xFF,0xD0,0x06,0xCE,0xA3,0xC5,0x4C,0x1F,0x8A,0xEE,0xA3,0xC5,0xAD,
  0x6B,0xCB,0x85,0x09,0xAD,0x6C,0xCB,0x85,0x0A,0xAD,0xA3,0xC5,0x85,0xA4,0xAD,0x5F,
  0xCB,0xC9,0x05,0x90,0x06,0x20,0x32,0x87,0x20,0xD4,0x88,0x4C,0xA7,0x89,0x01,0x00,
  0x00,0xFF,0x00,0xFF,0x01,0x00,0x01,0x01,0xFF,0xFF,0x01,0xFF,0x01,0xFF,0x00,0x01,
  0xFF,0x00,0x01,0x00,0x00,0xFF,0xFF,0x01,0xFF,0x01,0x01,0x01,0xFF,0xFF,0xFF,0x00,
  0x00,0x01,0x00,0x01,0xFF,0x00,0xFF,0xFF,0x01,0x01,0xFF,0x01,0xFF,0x01,0x00,0xFF,
  0x01,0x00,0xFF,0x00,0x00,0x01,0x01,0xFF,0x01,0xFF,0xFF,0xFF,0x01,0x01,0x20,0x2F,
  0x82,0xE0,0x08,0x90,0x05,0xA9,0x00,0x4C,0x4A,0x92,0x8A,0x0A,0x0A,0x0A,0xAA,0xA0,
  0x00,0xBD,0x3E,0x8A,0x99,0x57,0xCB,0xE8,0xC8,0xC0,0x08,0x30,0xF4,0x20,0x29,0x82,
  0x8E,0x04,0xC5,0x4C,0x57,0x82,0x20,0x4E,0x90,0x20,0xF0,0x88,0x84,0x09,0x85,0x0A,
  0x20,0x29,0x82,0x86,0xA4,0x20,0xED,0x88,0x84,0xAC,0x85,0xAD,0x20,0x9C,0x88,0x20,
  0x29,0x82,0xE0,0x00,0xD0,0x02,0xA2,0x01,0x8E,0x6C,0xCB,0x20,0xD1,0x8A,0x4C,0x57,
  0x82,0xA9,0x00,0x8D,0x6B,0xCB,0x20,0x6F,0x89,0x06,0xAC,0x26,0xAD,0x06,0xAC,0x26,
  0xAD,0x06,0xAC,0x26,0xAD,0x18,0xA9,0x00,0x65,0xAC,0x8D,0xA6,0xC5,0xA9,0xD0,0x65,
  0xAD,0x8D,0xA7,0xC5,0xA0,0x00,0x84,0xAC,0xC8,0x84,0xAD,0xA5,0x09,0x8D,0x99,0xC5,
  0xA5,0x0A,0x8D,0x9A,0xC5,0xA5,0xA4,0x8D,0xAA,0xC5,0xAC,0x6B,0xCB,0xA5,0xAC,0x29,
  0x07,0xAA,0xAD,0xA6,0xC5,0x85,0xFE,0xAD,0xA7,0xC5,0x85,0xFF,0xB1,0xFE,0x3D,0x4A,
  0x88,0xF0,0x03,0x20,0x32,0x87,0xE6,0xAC,0xEE,0x99,0xC5,0xD0,0x03,0xEE,0x9A,0xC5,
  0xA5,0xAC,0xC9,0x08,0xD0,0x2F,0xA9,0x00,0x85,0xAC,0x38,0xAD,0x99,0xC5,0xE9,0x08,
  0x8D,0x99,0xC5,0xB0,0x03,0xCE,0x9A,0xC5,0xA5,0xAD,0xCD,0x6C,0xCB,0xD0,0x07,0xEE,
  0x6B,0xCB,0xA9,0x00,0x85,0xAD,0xEE,0xAA,0xC5,0xE6,0xAD,0xAD,0x6B,0xCB,0xC9,0x08,
  0xD0,0x03,0x4C,0x68,0x89,0xAD,0x99,0xC5,0x85,0x09,0xAD,0x9A,0xC5,0x85,0x0A,0xAD,
  0xAA,0xC5,0x85,0xA4,0x4C,0x0A,0x8B,0xB1,0xA8,0x3D,0x4A,0x88,0xF0,0x04,0xA0,0x01,
  0x84,0x90,0x4C,0xD4,0x88,0xA9,0x00,0x8D,0x1B,0xCB,0x4C,0x03,0x86,0xA2,0x00,0xA0,
  0x00,0xCC,0x76,0xCB,0xF0,0x07,0xA5,0x66,0x91,0x20,0xC8,0xD0,0xF4,0xE8,0xEC,0x77,
  0xCB,0xF0,0x08,0xA0,0x00,0x20,0x7A,0xA2,0x4C,0x91,0x8B,0x60,0xA9,0x00,0x85,0xA6,
  0x20,0x03,0xAE,0xA5,0x21,0x48,0xA5,0x20,0x48,0x20,0x29,0x82,0x86,0x66,0x20,0x8D,
  0x8B,0x20,0x29,0x82,0x86,0x66,0x38,0x68,0xE9,0x00,0x85,0x20,0x68,0xED,0x88,0x02,
  0x85,0x21,0x18,0xA5,0x20,0x69,0x00,0x85,0x20,0xA5,0x21,0x69,0xD8,0x85,0x21,0x20,
  0x8D,0x8B,0x4C,0x57,0x82,0xA9,0x00,0x85,0xA6,0x20,0x03,0xAE,0x20,0x29,0x82,0x86,
  0x66,0x20,0x8D,0x8B,0x4C,0x57,0x82,0xA9,0x0A,0x85,0xA6,0x20,0x03,0xAE,0x20,0x29,
  0x82,0x86,0x66,0x20,0x8D,0x8B,0x4C,0x57,0x82,0xAC,0x15,0xC6,0xC0,0x09,0x90,0x05,
  0xA9,0x09,0x4C,0x4A,0x92,0xC8,0xA5,0x7A,0x99,0x01,0xC6,0xC8,0xA5,0x7B,0x99,0x01,
  0xC6,0x8C,0x15,0xC6,0x4C,0x03,0x86,0x20,0x4E,0x90,0xAD,0x15,0xC6,0xC9,0x02,0xB0,
  0x05,0xA9,0x08,0x4C,0x4A,0x92,0x20,0x3A,0x80,0x20,0x79,0x00,0xA5,0x61,0xF0,0x09,
  0xCE,0x15,0xC6,0xCE,0x15,0xC6,0x4C,0x57,0x82,0xAC,0x15,0xC6,0xB9,0x01,0xC6,0x85,
  0x7B,0x88,0xB9,0x01,0xC6,0x85,0x7A,0x4C,0x03,0x86,0xAE,0xFF,0xC5,0xE0,0x0A,0xF0,
  0x05,0xA9,0x00,0x4C,0x4A,0x92,0x20,0xA7,0x93,0x4C,0x03,0x86,0x20,0x2F,0x82,0x8E,
  0xFF,0xC5,0x4C,0x57,0x82,0x20,0x2F,0x82,0x8E,0xDA,0xC5,0x4C,0x57,0x82,0x20,0x4E,
  0x90,0x20,0x3A,0x80,0x20,0x79,0x00,0xC9,0x89,0xF0,0x05,0xA9,0xA7,0x20,0x5E,0x80,
  0xA5,0x61,0x8D,0xC9,0xC5,0xD0,0x1F,0xA0,0x00,0xB1,0x7A,0xF0,0x16,0xC9,0x64,0xD0,
  0x09,0x20,0x4E,0x90,0xB1,0x7A,0xC9,0x47,0xF0,0x06,0x20,0x4E,0x90,0x4C,0x99,0x8C,
  0x20,0x4E,0x90,0x4C,0x57,0x82,0xA0,0x00,0xB1,0x7A,0xC9,0x3A,0xB0,0x1F,0xC9,0x20,
  0xD0,0x06,0x20,0x4E,0x90,0x4C,0xB6,0x8C,0xC9,0x60,0xF0,0xF6,0x20,0x0D,0x80,0x20,
  0x16,0x80,0x20,0x55,0x80,0x48,0x20,0x27,0x83,0x68,0x4C,0xAE,0xA7,0xA5,0x7A,0xD0,
  0x02,0xC6,0x7B,0xC6,0x7A,0x6C,0x08,0x03,0x20,0x4E,0x90,0x20,0x0D,0x80,0x20,0x16,
  0x80,0x8D,0xFF,0x01,0x8C,0xFE,0x01,0x20,0x29,0x82,0x8E,0xAF,0xC5,0xA9,0x0A,0x8D,
  0xB0,0xC5,0x38,0xAD,0xFE,0x01,0xED,0xAF,0xC5,0x8D,0xFE,0x01,0xAD,0xFF,0x01,0xE9,
  0x00,0x8D,0xFF,0x01,0x4C,0x57,0x82,0x20,0x4E,0x90,0x20,0x0D,0x80,0x20,0x16,0x80,
  0x85,0x15,0x85,0x40,0x84,0x14,0x84,0x3F,0x20,0x70,0x80,0x38,0xA5,0x5F,0xE9,0x01,
  0xA4,0x60,0xB0,0x01,0x88,0x85,0x41,0x84,0x42,0x4C,0x57,0x82,0xA9,0x31,0x85,0x22,
  0xA9,0x0A,0x8D,0x99,0xC5,0x20,0x4B,0x8D,0x4C,0x78,0x8D,0x18,0xA5,0x2B,0x69,0x04,
  0x85,0x20,0xA5,0x2C,0x69,0x00,0x85,0x21,0xA0,0x00,0xB1,0x20,0xC9,0x64,0xD0,0x0A,
  0x20,0xEE,0x8D,0xB1,0x20,0xC5,0x22,0xD0,0x01,0x60,0xB1,0x20,0xF0,0x27,0xC9,0x22,
  0xF0,0x6C,0x20,0xEE,0x8D,0x4C,0x58,0x8D,0xC8,0xB1,0x7A,0xF0,0x0B,0xD1,0x20,0xD0,
  0x48,0xB1,0x20,0xF0,0x44,0x4C,0x78,0x8D,0xA0,0x00,0xA5,0x20,0x85,0x7A,0xA5,0x21,
  0x85,0x7B,0x4C,0x3F,0x90,0x20,0xEE,0x8D,0xB1,0x20,0xD0,0x05,0xA9,0x01,0x8D,0xDB,
  0xC5,0x20,0xEE,0x8D,0xB1,0x20,0xD0,0x2A,0xAD,0xDB,0xC5,0xC9,0x01,0xD0,0x23,0xAD,
  0x99,0xC5,0xC9,0x0A,0xF0,0x05,0x68,0x68,0x4C,0x57,0x82,0xA9,0x01,0x8D,0x99,0xC5,
  0xCE,0x2A,0xC6,0xCE,0x2A,0xC6,0x4C,0x4A,0x92,0x20,0xEE,0x8D,0x20,0x58,0x8D,0x4C,
  0x78,0x8D,0x20,0xEE,0x8D,0x20,0xEE,0x8D,0x20,0xEE,0x8D,0x4C,0x58,0x8D,0x20,0xEE,
  0x8D,0xA0,0x00,0xB1,0x20,0xF0,0xAE,0xC9,0x22,0xF0,0xED,0x4C,0xDE,0x8D,0xE6,0x20,
  0xD0,0x02,0xE6,0x21,0x60,0xA5,0x7A,0x48,0xA5,0x7B,0x48,0xA0,0x00,0xB1,0x7A,0xF0,
  0x06,0x20,0x4E,0x90,0x4C,0xFB,0x8D,0xAC,0x2A,0xC6,0xC0,0x09,0x90,0x05,0xA9,0x09,
  0x4C,0x4A,0x92,0xC8,0xA5,0x7A,0x99,0x16,0xC6,0xC8,0xA5,0x7B,0x99,0x16,0xC6,0x8C,
  0x2A,0xC6,0x68,0x85,0x7B,0x68,0x85,0x7A,0x4C,0x3C,0x8D,0xAC,0x2A,0xC6,0xC0,0x02,
  0xB0,0x05,0xA9,0x06,0x4C,0x4A,0x92,0xB9,0x16,0xC6,0x85,0x7B,0x88,0xB9,0x16,0xC6,
  0x85,0x7A,0x88,0x8C,0x2A,0xC6,0x4C,0x57,0x82,0x20,0x4E,0x90,0x20,0x4E,0x90,0xA9,
  0x8B,0x20,0x5E,0x80,0x20,0x3A,0x80,0x20,0x79,0x00,0xA5,0x61,0xD0,0x03,0x4C,0x57,
  0x82,0xA9,0x36,0x85,0x22,0xA5,0x7A,0x85,0x20,0xA5,0x7B,0x85,0x21,0x20,0x58,0x8D,
  0xA5,0x20,0x85,0x7A,0xA5,0x21,0x85,0x7B,0xCE,0x3F,0xC6,0xCE,0x3F,0xC6,0x4C,0x03,
  0x86,0xAC,0x3F,0xC6,0xC0,0x02,0xB0,0x05,0xA9,0x07,0x4C,0x4A,0x92,0xB9,0x2B,0xC6,
  0x85,0x7B,0x88,0xB9,0x2B,0xC6,0x85,0x7A,0x4C,0x03,0x86,0xA9,0x00,0x8D,0xE8,0xC5,
  0x20,0x3E,0xBF,0xA5,0x7A,0x8D,0xE6,0xC5,0xA5,0x7B,0x8D,0xE7,0xC5,0x20,0x58,0x90,
  0xA9,0x0A,0x8D,0xE8,0xC5,0xA0,0x00,0xB1,0x7A,0xF0,0x06,0x20,0x4E,0x90,0x4C,0xB5,
  0x8E,0x4C,0x57,0x82,0xA9,0x00,0x8D,0xE8,0xC5,0x4C,0x03,0x86,0xA5,0x7A,0x48,0xA5,
  0x7B,0x48,0x20,0x1E,0x83,0x20,0x0C,0xBF,0x20,0x27,0x83,0xEA,0xAD,0x4E,0xC7,0xD0,
  0x27,0x20,0x58,0x90,0x85,0xAD,0x86,0xAC,0x20,0x31,0x80,0x20,0xE4,0xFF,0xA0,0x0A,
  0x8C,0xE8,0xC5,0x8D,0xEA,0xC5,0xA0,0x00,0xB1,0xAC,0xCD,0xEA,0xC5,0xF0,0x10,0xC9,
  0x5D,0xF0,0x0C,0xC8,0xC4,0x69,0xD0,0xF0,0x68,0x85,0x7B,0x68,0x85,0x7A,0x60,0x20,
  0x1E,0x83,0x4C,0x27,0xBF,0x8D,0xEE,0xC5,0x68,0x8D,0xEF,0xC5,0x68,0x68,0x4C,0x57,
  0x82,0xAD,0xEE,0xC5,0x85,0x7B,0x20,0x1C,0xBF,0xEA,0xEA,0x20,0x27,0x83,0xA9,0x0A,
  0x8D,0xE8,0xC5,0x4C,0x3B,0x86,0xAC,0x3F,0xC6,0xC0,0x09,0x90,0x05,0xA9,0x09,0x4C,
  0x4A,0x92,0xC8,0xA5,0x7A,0x99,0x2B,0xC6,0xC8,0xA5,0x7B,0x99,0x2B,0xC6,0x8C,0x3F,
  0xC6,0x4C,0x03,0x86,0x20,0x2F,0x82,0x8E,0xF6,0xC5,0x4C,0x57,0x82,0x20,0x2F,0x82,
  0xA9,0x41,0x85,0x22,0x8D,0x99,0xC5,0x20,0x4B,0x8D,0xA5,0x20,0xD0,0x02,0xC6,0x21,
  0xC6,0x20,0xA9,0x00,0x91,0x20,0x20,0x72,0x8D,0xA9,0x3A,0x91,0x20,0x4C,0x6A,0x8F,
  0xA9,0x00,0x8D,0xF9,0xC5,0x20,0x4E,0x90,0xA5,0x7A,0x8D,0xF7,0xC5,0xA5,0x7B,0x8D,
  0xF8,0xC5,0xA9,0x0A,0x8D,0xF9,0xC5,0xA0,0x00,0xB1,0x7A,0xF0,0x06,0x20,0x4E,0x90,
  0x4C,0x97,0x8F,0x4C,0x57,0x82,0xA9,0x00,0x8D,0xF9,0xC5,0x4C,0x03,0x86,0x8E,0xFA,
  0xC5,0x20,0x27,0x83,0xAD,0xF9,0xC5,0xC9,0x0A,0xF0,0x06,0xAE,0xFA,0xC5,0x4C,0x8B,
  0xE3,0xAD,0xF8,0xC5,0x85,0x7B,0xAD,0xF7,0xC5,0x85,0x7A,0xA5,0x39,0x8D,0x1F,0xC5,
  0xA5,0x3A,0x8D,0x20,0xC5,0xA9,0x00,0x8D,0x15,0xC6,0x8D,0x2A,0xC6,0x8D,0x3F,0xC6,
  0x4C,0x57,0x82,0x20,0x27,0x83,0x4C,0xBB,0x8F,0xA9,0x04,0x85,0x20,0xA5,0x2C,0x85,
  0x21,0xA0,0x00,0x20,0xEE,0x8D,0xB1,0x20,0xD0,0xF9,0x20,0xEE,0x8D,0xA5,0x20,0x91,
  0x2B,0xC8,0xA5,0x21,0x91,0x2B,0x88,0xA9,0x01,0x85,0x2D,0xA2,0x03,0x20,0xEE,0x8D,
  0xB1,0x20,0xD0,0xF7,0xCA,0xD0,0xF6,0x20,0xEE,0x8D,0xA5,0x20,0x85,0x31,0xA5,0x21,
  0x85,0x32,0xA2,0x03,0xB5,0x2F,0x95,0x2D,0xCA,0x10,0xF9,0x4C,0x03,0x86,0x20,0x4E,
  0x90,0x20,0x27,0x83,0x20,0x4E,0x90,0xAD,0xC9,0xC5,0x85,0x61,0x4C,0x90,0x8C,0xA0,
  0x00,0xB1,0x7A,0xF0,0x06,0x20,0x4E,0x90,0x4C,0x41,0x90,0x4C,0x57,0x82,0xE6,0x7A,
  0xD0,0x02,0xE6,0x7B,0x60,0x20,0x4E,0x90,0x20,0x3A,0x80,0x20,0x43,0x80,0x85,0x69,
  0xA5,0x23,0xA6,0x22,0x60,0x99,0x86,0x01,0x87,0x71,0x88,0xEC,0x84,0xE4,0x8B,0xF6,
  0x8B,0xAB,0x8B,0x38,0xBC,0x7D,0x8A,0x79,0x89,0xA5,0x8A,0x84,0x8B,0xD2,0xAD,0x47,
  0x92,0xD2,0xAD,0x47,0x92,0xD2,0xAD,0xD2,0xAD,0xD2,0xAD,0xD2,0xAD,0xD2,0xAD,0xD2,
  0xAD,0xD2,0xAD,0xD2,0xAD,0x33,0x9B,0x80,0x9B,0x11,0x9C,0xB2,0x9C,0x21,0x9D,0xA0,
  0x9D,0xD7,0x9D,0x08,0x8C,0x5F,0x9F,0x5F,0x9F,0x8E,0x9F,0xC0,0x9F,0x2F,0xA0,0x3E,
  0xA0,0x6D,0xB1,0x47,0x92,0x26,0x8C,0x12,0xA2,0x12,0xA2,0xFA,0xB4,0xFA,0xB4,0x12,
  0xA2,0x16,0x8D,0x16,0x8D,0x3E,0x90,0x3B,0x8D,0xF4,0x8D,0x2A,0x8E,0x48,0x8E,0x80,
  0x8E,0x9A,0x8E,0xC3,0x8E,0x20,0x8F,0x35,0x8F,0x53,0x8F,0x5C,0x8F,0x5C,0x8F,0x5C,
  0x8F,0x5C,0x8F,0x5C,0x8F,0x02,0x86,0x34,0xA2,0x7F,0x8F,0xA5,0x8F,0xCA,0xA1,0x2D,
  0x90,0x3E,0x90,0x59,0x8C,0x6B,0x8C,0x8F,0x9B,0x7D,0x9E,0xC4,0xA0,0x86,0xA4,0x74,
  0x8C,0xE7,0x8C,0xE8,0x8F,0x47,0x92,0x47,0x92,0x47,0x92,0x47,0x92,0x47,0x92,0x47,
  0x92,0x47,0x92,0x47,0x92,0x47,0x92,0x47,0x92,0x47,0x92,0x47,0x92,0x47,0x92,0x47,
  0x92,0x47,0x92,0x47,0x92,0x47,0x92,0xF2,0xA7,0x29,0xA9,0x67,0xA9,0x76,0xA9,0xA3,
  0xA9,0xF2,0xA9,0x17,0xAA,0x26,0xAA,0xC2,0xAC,0x37,0xAA,0x51,0x81,0xF8,0xB2,0x45,
  0xB3,0x92,0xB3,0x74,0xB4,0xA6,0xB4,0xC4,0xB4,0x07,0xB6,0xF7,0xB6,0x4F,0xB7,0xA1,
  0x88,0x31,0xBB,0x8D,0x83,0xC8,0x83,0xF7,0xBC,0x41,0xBD,0x41,0xBD,0x5D,0xBD,0x47,
  0x92,0xE2,0x8F,0x50,0x52,0x4F,0x43,0x20,0x4E,0x4F,0x54,0x20,0x46,0x4F,0x55,0x4E,
  0x44,0x00,0x49,0x4E,0x53,0x45,0x52,0x54,0x20,0x50,0x41,0x52,0x41,0x4D,0x45,0x54,
  0x45,0x52,0x20,0x54,0x4F,0x4F,0x20,0x4C,0x41,0x52,0x47,0x45,0x00,0x43,0x52,0x45,
  0x41,0x54,0x45,0x44,0x20,0x53,0x54,0x52,0x49,0x4E,0x47,0x20,0x54,0x4F,0x4F,0x20,
  0x4C,0x41,0x52,0x47,0x45,0x00,0x4E,0x4F,0x54,0x20,0x42,0x49,0x4E,0x41,0x52,0x59,
  0x20,0x43,0x48,0x41,0x52,0x41,0x43,0x54,0x45,0x52,0x00,0x4E,0x4F,0x54,0x20,0x48,
  0x45,0x58,0x20,0x43,0x48,0x41,0x52,0x41,0x43,0x54,0x45,0x52,0x00,0x45,0x4E,0x44,
  0x20,0x50,0x52,0x4F,0x43,0x20,0x57,0x49,0x54,0x48,0x4F,0x55,0x54,0x20,0x45,0x58,
  0x45,0x43,0x00,0x45,0x4E,0x44,0x20,0x4C,0x4F,0x4F,0x50,0x20,0x57,0x49,0x54,0x48,
  0x4F,0x55,0x54,0x20,0x4C,0x4F,0x4F,0x50,0x00,0x55,0x4E,0x54,0x49,0x4C,0x20,0x57,
  0x49,0x54,0x48,0x4F,0x55,0x54,0x20,0x52,0x45,0x50,0x45,0x41,0x54,0x00,0x53,0x54,
  0x41,0x43,0x4B,0x20,0x54,0x4F,0x4F,0x20,0x4C,0x41,0x52,0x47,0x45,0x00,0x54,0x4F,
  0x4F,0x20,0x46,0x45,0x57,0x20,0x4C,0x49,0x4E,0x45,0x53,0x00,0x42,0x41,0x44,0x20,
  0x43,0x48,0x41,0x52,0x20,0x46,0x4F,0x52,0x20,0x41,0x20,0x4D,0x4F,0x42,0x00,0x42,
  0x41,0x44,0x20,0x4D,0x4F,0x44,0x45,0x00,0xA9,0x00,0x0A,0xA8,0xB9,0x79,0x92,0x85,
  0x21,0xB9,0x78,0x92,0x85,0x20,0x20,0x27,0x83,0x20,0xD4,0x88,0xA9,0x3F,0x20,0xD2,
  0xFF,0xA0,0x00,0xB1,0x20,0xF0,0x06,0x20,0xD2,0xFF,0xC8,0xD0,0xF6,0xA4,0x3A,0xC8,
  0xF0,0x03,0x20,0xC2,0xBD,0x4C,0x86,0xE3,0x3F,0x92,0x63,0x91,0x72,0x91,0x8D,0x91,
  0xA6,0x91,0xBB,0x91,0xCD,0x91,0xE3,0x91,0xF9,0x91,0x0E,0x92,0x1E,0x92,0x2C,0x92,
  0x3F,0x92,0xAD,0x00,0xDC,0x85,0x20,0x29,0xEF,0x8D,0x6E,0xCB,0xA2,0x00,0xBD,0xD0,
  0x92,0xCD,0x6E,0xCB,0xF0,0x0D,0xE8,0xE0,0x08,0xD0,0xF3,0xA9,0x00,0x8D,0x6E,0xCB,
  0x4C,0xB7,0x92,0xE8,0x8E,0x6E,0xCB,0xA5,0x20,0x29,0x90,0xD0,0x08,0xAD,0x6E,0xCB,
  0x09,0x80,0x8D,0x6E,0xCB,0xAC,0x6E,0xCB,0xA2,0x00,0x20,0x4E,0x90,0x4C,0x8C,0x96,
  0x6E,0x66,0x67,0x65,0x6D,0x69,0x6B,0x6A,0x20,0x73,0x00,0x20,0xFA,0xAE,0x20,0xDB,
  0x88,0x20,0xF7,0xAE,0xA5,0x65,0xD0,0x06,0xAC,0x19,0xD4,0x4C,0xF1,0x92,0xAC,0x1A,
  0xD4,0x84,0x63,0xA2,0x00,0x4C,0x8C,0x96,0x20,0x0C,0x93,0xAC,0x02,0xC5,0x4C,0x04,
  0x93,0xAC,0x03,0xC5,0xA2,0x00,0x20,0x4E,0x90,0x4C,0x8C,0x96,0xAD,0x19,0xD0,0x29,
  0x08,0xF0,0xF9,0xAD,0x13,0xD0,0x8D,0x02,0xC5,0xAD,0x14,0xD0,0x8D,0x03,0xC5,0xAD,
  0x19,0xD0,0x29,0xF7,0x8D,0x19,0xD0,0x60,0xA5,0x39,0x85,0x63,0xA5,0x3A,0x85,0x62,
  0xA5,0x7B,0xC9,0x03,0xB0,0x01,0x60,0xA2,0x90,0x38,0x20,0x88,0x80,0x20,0xB5,0x80,
  0x20,0x56,0x93,0x20,0xF8,0x93,0xA2,0x00,0xBD,0x00,0x01,0xF0,0x08,0x49,0x80,0x9D,
  0x57,0xC5,0xE8,0xD0,0xF3,0x60,0x78,0xA9,0xC5,0x85,0x24,0xA9,0x33,0x85,0x23,0xA9,
  0xC5,0x85,0x21,0xA9,0x39,0x85,0x20,0xA0,0x00,0xA2,0x00,0xB1,0x20,0x91,0x23,0xC8,
  0xC0,0x05,0xD0,0xF7,0x18,0xA5,0x20,0x69,0x06,0x85,0x20,0xA5,0x21,0x69,0x00,0x85,
  0x21,0x18,0xA5,0x23,0x69,0x06,0x85,0x23,0xA5,0x24,0x69,0x00,0x85,0x24,0xE8,0xA0,
  0x00,0xE0,0x06,0xD0,0xD6,0xA9,0xA0,0x91,0x23,0xC8,0xC0,0x05,0xD0,0xF9,0x58,0x60,
  0xA5,0x7B,0xC9,0x03,0xB0,0x01,0x60,0x20,0xF8,0x93,0x78,0xA9,0xC5,0x85,0x24,0xA9,
  0x33,0x85,0x23,0xA9,0x04,0x85,0x21,0xA9,0x22,0x85,0x20,0xA0,0x00,0xA2,0x00,0xA9,
  0xA3,0x91,0x20,0xB1,0x23,0xC8,0x91,0x20,0xC0,0x05,0xD0,0xF7,0x20,0x7A,0xA2,0x18,
  0xA5,0x23,0x69,0x06,0x85,0x23,0xA5,0x24,0x69,0x00,0x85,0x24,0xE8,0xA0,0x00,0xE0,
  0x06,0xD0,0xDC,0x58,0xAD,0x8D,0x02,0xC9,0x02,0xD0,0x0C,0xA2,0x00,0xA0,0x00,0xE8,
  0xD0,0xFD,0xC8,0xC0,0x82,0xD0,0xF8,0x60,0x78,0xA9,0xD8,0x85,0x21,0xA9,0x22,0x85,
  0x20,0xA0,0x00,0xA2,0x00,0xA9,0x01,0x91,0x20,0xC8,0xC0,0x06,0xD0,0xF7,0x20,0x7A,
  0xA2,0xE8,0xA0,0x00,0xE0,0x06,0xD0,0xED,0x58,0x60,0xAD,0x88,0xCB,0xC9,0x00,0xF0,
  0x1C,0xCE,0x84,0xCB,0xAD,0x84,0xCB,0xC9,0x00,0xF0,0x01,0x60,0xCE,0x88,0xCB,0xAD,
  0x88,0xCB,0xC9,0x00,0xF0,0x07,0xAD,0x86,0xCB,0x8D,0x84,0xCB,0x60,0xAD,0x99,0xCB,
  0x85,0xFC,0xAD,0x9A,0xCB,0x85,0xFD,0xAC,0x9B,0xCB,0xB1,0xFC,0xC9,0x85,0x90,0x18,
  0xC9,0x8D,0xB0,0x14,0x38,0xB1,0xFC,0xE9,0x84,0x8D,0x84,0xCB,0x8D,0x86,0xCB,0xAD,
  0x94,0xCB,0x8D,0x88,0xCB,0x4C,0x8D,0x94,0xC9,0x93,0xD0,0x03,0x4C,0xA9,0x94,0x8C,
  0x9D,0xCB,0x20,0x5A,0x95,0xAC,0x8A,0xCB,0x84,0xFE,0xAC,0x8B,0xCB,0x84,0xFF,0xA0,
  0x01,0x91,0xFE,0x8A,0x88,0x91,0xFE,0x20,0xD3,0x95,0xAC,0x9D,0xCB,0xCC,0x9C,0xCB,
  0xB0,0x05,0xC8,0x8C,0x9B,0xCB,0x60,0xA9,0x00,0x8D,0x9B,0xCB,0x8D,0x8F,0xCB,0x60,
  0x4C,0x27,0x95,0x4C,0x38,0x95,0x4C,0x49,0x95,0xC8,0xB1,0xFC,0xA2,0x00,0x8E,0x88,
  0xCB,0xC9,0x31,0xF0,0xEB,0xC9,0x32,0xF0,0xEA,0xC9,0x33,0xF0,0xE9,0xC9,0x47,0xF0,
  0x47,0x48,0x8C,0x9D,0xCB,0xAC,0x8A,0xCB,0x84,0xFE,0xAC,0x8B,0xCB,0x84,0xFF,0xAC,
  0x9D,0xCB,0x68,0xC9,0x54,0xD0,0x13,0x98,0x48,0xAC,0xA6,0xCB,0xB9,0x48,0xC6,0x49,
  0x02,0xA0,0x04,0x91,0xFE,0x68,0xA8,0x4C,0x8D,0x94,0xC9,0x43,0xD0,0x0A,0x98,0x48,
  0xAC,0xA6,0xCB,0xA9,0x00,0x4C,0xE1,0x94,0xC9,0x52,0xD0,0x09,0xA0,0x00,0x8C,0x9B,
  0xCB,0x8C,0x88,0xCB,0x60,0x4C,0x8D,0x94,0x8C,0x9D,0xCB,0xAC,0x8A,0xCB,0x84,0xFE,
  0xAC,0x8B,0xCB,0x84,0xFF,0xAC,0xA6,0xCB,0xB9,0x48,0xC6,0x29,0xFE,0xA0,0x04,0x91,
  0xFE,0xAC,0x9D,0xCB,0x4C,0x8D,0x94,0xA9,0xD4,0x8D,0x8B,0xCB,0xA9,0x00,0x8D,0x8A,
  0xCB,0xA9,0x00,0x8D,0xA6,0xCB,0xF0,0xCD,0xA9,0xD4,0x8D,0x8B,0xCB,0xA9,0x07,0x8D,
  0x8A,0xCB,0xA9,0x01,0x8D,0xA6,0xCB,0xD0,0xBC,0xA9,0xD4,0x8D,0x8B,0xCB,0xA9,0x0E,
  0x8D,0x8A,0xCB,0xA9,0x02,0x8D,0xA6,0xCB,0xD0,0xAB,0xA0,0x00,0xD9,0x97,0x95,0xF0,
  0x09,0xC8,0xC0,0x14,0xD0,0xF6,0xA9,0x00,0xAA,0x60,0xB9,0xBF,0x95,0x8D,0x9C,0xC5,
  0xB9,0xAB,0x95,0x8D,0x9B,0xC5,0xAC,0x9D,0xCB,0xC8,0x8C,0x9D,0xCB,0xB1,0xFC,0xE9,
  0x30,0xAA,0xE0,0x00,0xF0,0x0A,0x0E,0x9B,0xC5,0x2E,0x9C,0xC5,0xCA,0x4C,0x82,0x95,
  0xAD,0x9C,0xC5,0xAE,0x9B,0xC5,0x60,0x43,0xC3,0x44,0xC4,0x45,0x46,0xC6,0x47,0xC7,
  0x41,0xC1,0x42,0x5A,0xAC,0xB1,0xBB,0xA5,0xB0,0xBF,0xBC,0x12,0x23,0x34,0x46,0x5A,
  0x6E,0x84,0x9B,0xB3,0xCD,0xE9,0x06,0x00,0x23,0x46,0x5A,0x84,0xB3,0xE9,0x06,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x02,0x00,0x01,0x01,0x01,0x01,
  0x01,0x01,0x02,0xAC,0xA6,0xCB,0xB9,0xA2,0xCB,0xA0,0x05,0x91,0xFE,0xAC,0xA6,0xCB,
  0xB9,0x9E,0xCB,0xA0,0x06,0x91,0xFE,0xAC,0xA6,0xCB,0xB9,0x48,0xC6,0x29,0xFE,0xA0,
  0x04,0x91,0xFE,0xA0,0x04,0x09,0x01,0x91,0xFE,0x60,0xAE,0xBA,0xC5,0xAD,0x1D,0xD0,
  0x1D,0xC0,0x9D,0x8D,0x1D,0xD0,0x60,0xAE,0xBA,0xC5,0xAD,0x17,0xD0,0x1D,0xC0,0x9D,
  0x8D,0x17,0xD0,0x60,0xAE,0xBA,0xC5,0xAD,0x1D,0xD0,0x3D,0xC8,0x9D,0x8D,0x1D,0xD0,
  0x60,0xAE,0xBA,0xC5,0xAD,0x17,0xD0,0x3D,0xC8,0x9D,0x8D,0x17,0xD0,0x60,0x20,0xFD,
  0xAE,0x4C,0x9E,0xB7,0xAD,0x98,0xC5,0x48,0x20,0x58,0x90,0x8D,0x7F,0xC5,0x68,0x8D,
  0x98,0xC5,0xAD,0x7F,0xC5,0x60,0xA9,0x00,0x85,0x0D,0x20,0x73,0x00,0xB0,0x03,0x4C,
  0x79,0x80,0xC9,0x24,0xF0,0x40,0xC9,0x25,0xF0,0x22,0xC9,0x64,0xF0,0x03,0x4C,0x7C,
  0x80,0x4C,0x9B,0x97,0xC9,0x30,0x90,0x0C,0xC9,0x47,0xB0,0x08,0xC9,0x3A,0x90,0x0B,
  0xC9,0x41,0xB0,0x07,0x68,0x68,0xA9,0x05,0x4C,0x4A,0x92,0x60,0x20,0x73,0x00,0x20,
  0x1E,0x83,0x20,0x51,0xA0,0x20,0x27,0x83,0xA4,0xA8,0xA2,0x00,0x84,0x63,0x86,0x62,
  0xA2,0x90,0x38,0x4C,0x49,0xBC,0x20,0x73,0x00,0x20,0x9F,0x96,0x4C,0xDD,0x96,0xA0,
  0x00,0xB1,0x7A,0x20,0x64,0x96,0x8D,0x21,0xC5,0x20,0x73,0x00,0xB1,0x7A,0x20,0x64,
  0x96,0xA8,0xAD,0x21,0xC5,0x20,0xE7,0x96,0x85,0xA9,0x20,0x73,0x00,0xA0,0x00,0xB1,
  0x7A,0x20,0x64,0x96,0x8D,0x21,0xC5,0x20,0x73,0x00,0xB1,0x7A,0x20,0x64,0x96,0xA8,
  0xAD,0x21,0xC5,0x20,0xE7,0x96,0x85,0x63,0xA5,0xA9,0x85,0x62,0x60,0x20,0x73,0x00,
  0xA2,0x90,0x38,0x20,0x49,0xBC,0x60,0x48,0x98,0x20,0xFB,0x96,0x8D,0x52,0xCB,0x68,
  0x20,0xFB,0x96,0x0A,0x0A,0x0A,0x0A,0x0D,0x52,0xCB,0x60,0x38,0xE9,0x30,0xC9,0x0A,
  0x90,0x02,0xE9,0x07,0x60,0x20,0x73,0x00,0x20,0xFA,0xAE,0x20,0xDB,0x88,0x84,0xAA,
  0x85,0xAB,0x20,0xFD,0xAE,0x20,0xDB,0x88,0x20,0xF7,0xAE,0xA5,0x65,0x45,0xAA,0xA8,
  0xA5,0x64,0x45,0xAB,0xAA,0x4C,0x8C,0x96,0x20,0x73,0x00,0x20,0xFA,0xAE,0x20,0x34,
  0x96,0xA4,0x69,0x8C,0x21,0xC5,0x8D,0xAA,0xCB,0x8E,0xA9,0xCB,0x20,0xFD,0xAE,0x20,
  0xDB,0x88,0x20,0xF7,0xAE,0xA6,0x65,0x8E,0xAC,0xCB,0xA0,0x00,0x8C,0x22,0xC5,0xA0,
  0x00,0xEE,0x22,0xC5,0xC8,0xCC,0x21,0xC5,0xD0,0xF7,0xCA,0xD0,0xF2,0xAD,0x22,0xC5,
  0x20,0x7D,0xB4,0xAD,0xAA,0xCB,0x85,0x21,0xAD,0xA9,0xCB,0x85,0x20,0xAE,0xAC,0xCB,
  0xA0,0x00,0x8C,0x1A,0xC5,0xB1,0x20,0x8C,0x1B,0xC5,0xAC,0x1A,0xC5,0x91,0x62,0xC8,
  0x8C,0x1A,0xC5,0xAC,0x1B,0xC5,0xC8,0xCC,0x21,0xC5,0xD0,0xE9,0xA0,0x00,0x8C,0x22,
  0xC5,0xCA,0xD0,0xE1,0xA9,0xFF,0x85,0x0D,0x4C,0xCA,0xB4,0x20,0x73,0x00,0xA0,0x00,
  0xB1,0x7A,0xC9,0x55,0xD0,0x03,0x4C,0x28,0x97,0xC9,0x5A,0xD0,0x03,0x4C,0x05,0x97,
  0xC9,0x51,0xD0,0x03,0x4C,0x92,0x92,0xC9,0x0E,0xD0,0x4C,0x20,0x73,0x00,0xA5,0x7A,
  0x48,0xA5,0x7B,0x48,0xA9,0x00,0x85,0x62,0x85,0x63,0xA2,0x90,0x38,0x20,0x49,0xBC,
  0x20,0x0C,0xBC,0x20,0xFA,0xAE,0x20,0x9E,0xAD,0xA5,0x66,0x85,0xAA,0x30,0x03,0x4C,
  0xE5,0x97,0x20,0xB4,0xBF,0x20,0xF7,0xAE,0x20,0x1B,0xBC,0x20,0x0C,0xBC,0x20,0xCC,
  0xBC,0x68,0x68,0x20,0x53,0xB8,0xA9,0x00,0x85,0x0D,0xA5,0xAA,0x30,0x03,0x4C,0x04,
  0x98,0x20,0xB4,0xBF,0x4C,0x1B,0xBC,0xC9,0x10,0xF0,0x03,0x4C,0x6B,0x98,0x20,0x73,
  0x00,0x20,0xFA,0xAE,0x20,0x34,0x96,0x85,0xA8,0x86,0xA7,0xA5,0x69,0x8D,0xB5,0xCB,
  0x20,0xFD,0xAE,0x20,0x34,0x96,0x85,0xAA,0x86,0xA9,0xA5,0x69,0x8D,0xB2,0xCB,0x20,
  0xF7,0xAE,0xA0,0x00,0x8C,0xAF,0xCB,0xB1,0xA7,0xD1,0xA9,0xD0,0x16,0xC8,0xCC,0xB5,
  0xCB,0xD0,0xF4,0xEE,0xAF,0xCB,0xAD,0xAF,0xCB,0xA8,0xA9,0x00,0x85,0x0D,0xA2,0x00,
  0x4C,0x8C,0x96,0xA0,0x00,0xE6,0xA9,0xD0,0x02,0xE6,0xAA,0xEE,0xAF,0xCB,0xAD,0xAF,
  0xCB,0xCD,0xB2,0xCB,0xD0,0xD1,0xA9,0x00,0x4C,0x49,0x98,0xC9,0x5B,0xF0,0x03,0x4C,
  0x49,0x99,0x20,0x73,0x00,0x20,0xFA,0xAE,0x20,0x34,0x96,0x85,0xA8,0x86,0xA7,0xA5,
  0x69,0x8D,0xB5,0xCB,0x20,0xFD,0xAE,0x20,0x34,0x96,0x85,0xAA,0x86,0xA9,0xA5,0x69,
  0x8D,0xB2,0xCB,0x20,0x2E,0x96,0x8E,0x21,0xC5,0x20,0xF7,0xAE,0x18,0xAD,0xB2,0xCB,
  0x6D,0xB5,0xCB,0x90,0x0A,0xA9,0x03,0x4C,0x4A,0x92,0xA9,0x02,0x4C,0x4A,0x92,0x20,
  0x7D,0xB4,0xA0,0x00,0xCC,0x21,0xC5,0xF0,0x0C,0xB1,0xA9,0x91,0x62,0xC8,0xCC,0xB2,
  0xCB,0xF0,0xE7,0xD0,0xEF,0x8C,0x22,0xC5,0x8C,0xAC,0xCB,0xA0,0x00,0x8C,0xA9,0xCB,
  0xB1,0xA7,0xEE,0xA9,0xCB,0xAC,0x22,0xC5,0x91,0x62,0xEE,0x22,0xC5,0xAC,0xA9,0xCB,
  0xCC,0xB5,0xCB,0xD0,0xEB,0xAC,0xAC,0xCB,0xB1,0xA9,0xEE,0xAC,0xCB,0xAC,0x22,0xC5,
  0x91,0x62,0xEE,0x22,0xC5,0xAC,0xAC,0xCB,0xCC,0xB2,0xCB,0xD0,0xEB,0xA9,0xFF,0x85,
  0x0D,0x4C,0xCA,0xB4,0xA9,0x00,0x8D,0xC0,0xCB,0x8D,0xC1,0xCB,0xAD,0xB9,0xCB,0x0D,
  0xBA,0xCB,0xD0,0x07,0x68,0x68,0xA2,0x14,0x6C,0x00,0x03,0xA2,0x10,0x2E,0xBE,0xCB,
  0x2E,0xBF,0xCB,0x2E,0xC0,0xCB,0x2E,0xC1,0xCB,0x38,0xAD,0xC0,0xCB,0xED,0xB9,0xCB,
  0xA8,0xAD,0xC1,0xCB,0xED,0xBA,0xCB,0x90,0x06,0x8C,0xC0,0xCB,0x8D,0xC1,0xCB,0xCA,
  0xD0,0xDB,0x2E,0xBE,0xCB,0x2E,0xBF,0xCB,0x60,0xC9,0x53,0xD0,0x2A,0x20,0x73,0x00,
  0x20,0xFA,0xAE,0x20,0xDB,0x88,0x8D,0xBF,0xCB,0x8C,0xBE,0xCB,0x20,0xFD,0xAE,0x20,
  0xDB,0x88,0x8D,0xBA,0xCB,0x8C,0xB9,0xCB,0x20,0xF7,0xAE,0x20,0x04,0x99,0xAC,0xBE,
  0xCB,0xAE,0xBF,0xCB,0x4C,0x8C,0x96,0xC9,0x52,0xD0,0x2A,0x20,0x73,0x00,0x20,0xFA,
  0xAE,0x20,0xDB,0x88,0x8D,0xBF,0xCB,0x8C,0xBE,0xCB,0x20,0xFD,0xAE,0x20,0xDB,0x88,
  0x8D,0xBA,0xCB,0x8C,0xB9,0xCB,0x20,0xF7,0xAE,0x20,0x04,0x99,0xAC,0xC0,0xCB,0xAE,
  0xC1,0xCB,0x4C,0x8C,0x96,0xC9,0x59,0xD0,0x0E,0x20,0x4E,0x90,0x38,0x20,0xF0,0xFF,
  0x8A,0xA8,0xA2,0x00,0x4C,0x8C,0x96,0xC9,0x58,0xD0,0x31,0x20,0x73,0x00,0x20,0xFA,
  0xAE,0x20,0xDB,0x88,0x84,0x09,0x85,0x0A,0x20,0xFD,0xAE,0x20,0xDB,0x88,0x84,0xA4,
  0x20,0xF7,0xAE,0xA9,0x0B,0x85,0xF7,0xA9,0x00,0x85,0x90,0x20,0x1E,0x83,0x20,0x32,
  0x87,0xA2,0x00,0xA4,0x90,0x20,0x27,0x83,0x58,0x4C,0x8C,0x96,0xC9,0x57,0xF0,0x03,
  0x4C,0x64,0x9A,0x20,0x73,0x00,0x20,0xFA,0xAE,0x20,0x34,0x96,0x85,0xA8,0x86,0xA7,
  0xA5,0x69,0x8D,0xB5,0xCB,0x20,0xFD,0xAE,0x20,0x34,0x96,0x85,0xAA,0x86,0xA9,0xA5,
  0x69,0x8D,0xB2,0xCB,0x20,0x2E,0x96,0x8E,0x21,0xC5,0x20,0xF7,0xAE,0x18,0xAD,0xB5,
  0xCB,0x6D,0x21,0xC5,0xCD,0xB2,0xCB,0x90,0x08,0x4C,0x34,0x9A,0xA9,0x03,0x4C,0x4A,
  0x92,0xAD,0xB2,0xCB,0x20,0x7D,0xB4,0xA0,0x00,0xB1,0xA9,0x91,0x62,0xC8,0xCC,0xB2,
  0xCB,0xD0,0xF6,0xA0,0x00,0x8C,0x22,0xC5,0xB1,0xA7,0xEE,0x22,0xC5,0xAC,0x21,0xC5,
  0x91,0x62,0xEE,0x21,0xC5,0xAC,0x22,0xC5,0xCC,0xB5,0xCB,0xD0,0xEB,0xA9,0xFF,0x85,
  0x0D,0x4C,0xCA,0xB4,0xC9,0x56,0xD0,0x0B,0x20,0x4E,0x90,0xAC,0xD4,0xC5,0xA2,0x00,
  0x4C,0x8C,0x96,0xC9,0x5C,0xD0,0x03,0x4C,0xD8,0x92,0xC9,0x5D,0xD0,0x03,0x4C,0xF8,
  0x92,0xC9,0x5F,0xD0,0x03,0x4C,0x01,0x93,0xC9,0x61,0xD0,0x0A,0x20,0x4E,0x90,0xA2,
  0xD0,0xA0,0x00,0x4C,0x8C,0x96,0xC9,0x60,0xD0,0x0A,0x20,0x4E,0x90,0xA2,0xD4,0xA0,
  0x00,0x4C,0x8C,0x96,0xC9,0x28,0xD0,0x03,0x4C,0xFF,0x9D,0xC9,0x7C,0xD0,0x57,0x20,
  0x4E,0x90,0x20,0xFA,0xAE,0x20,0xDB,0x88,0x8C,0x21,0xC5,0xAD,0x17,0xC5,0xD0,0x33,
  0x20,0xFD,0xAE,0x20,0xDB,0x88,0x8C,0x22,0xC5,0x20,0xF7,0xAE,0xAC,0x21,0xC5,0xAD,
  0x16,0xC5,0x19,0xC0,0x9D,0xCD,0x16,0xC5,0xD0,0x15,0xAC,0x22,0xC5,0xAD,0x16,0xC5,
  0x19,0xC0,0x9D,0xCD,0x16,0xC5,0xD0,0x07,0xA0,0x00,0xA2,0x00,0x4C,0x8C,0x96,0xA0,
  0x01,0xD0,0xF7,0x20,0xF7,0xAE,0xAC,0x21,0xC5,0xAD,0x16,0xC5,0x19,0xC0,0x9D,0xCD,
  0x16,0xC5,0xD0,0xEB,0xF0,0xE2,0x20,0x4E,0x90,0xA0,0x00,0xB1,0x7A,0xC9,0x4E,0xF0,
  0x1C,0xC9,0x4C,0xF0,0x03,0x4C,0x57,0x82,0x20,0x4E,0x90,0xB1,0x7A,0xC9,0x4E,0xD0,
  0xF4,0xAC,0x1F,0xC5,0xAE,0x20,0xC5,0x20,0x4E,0x90,0x4C,0x8C,0x96,0xAC,0xFA,0xC5,
  0xA2,0x00,0xF0,0xF3,0xA9,0x2C,0x8D,0xAE,0xC5,0xA9,0x9F,0x8D,0x2F,0xC5,0xA9,0x00,
  0x8D,0x30,0xC5,0x20,0x14,0xA0,0xA0,0x00,0xA2,0x00,0x8A,0xA5,0xAA,0x99,0x00,0xC0,
  0x99,0xF9,0xC0,0x99,0xF3,0xC1,0x99,0xED,0xC2,0xE8,0xC8,0xC0,0xFB,0xD0,0xEB,0x20,
  0x29,0x82,0x86,0xAA,0xA0,0x00,0xA5,0xAA,0x99,0x00,0xD8,0x99,0xF9,0xD8,0x99,0xF3,
  0xD9,0x99,0xED,0xDA,0xC8,0xC0,0xFB,0xD0,0xED,0xA9,0x18,0x8D,0x16,0xD0,0x4C,0x57,
  0x82,0x20,0x2F,0x82,0x8E,0x20,0xD0,0x20,0x29,0x82,0x8E,0x21,0xD0,0x4C,0x57,0x82,
  0x20,0x4E,0x90,0x20,0xCC,0xFF,0xA9,0x01,0x20,0xC3,0xFF,0x20,0xCC,0xFF,0x20,0x58,
  0x90,0x86,0xAE,0x85,0xAF,0xA0,0x00,0xB1,0xAE,0xC9,0x24,0xF0,0x05,0xA9,0x00,0x4C,
  0x4A,0x92,0xA4,0xAF,0xA5,0x69,0x20,0xBD,0xFF,0xA9,0x01,0xA2,0x08,0xA0,0x00,0x20,
  0xBA,0xFF,0x20,0xC0,0xFF,0xA2,0x01,0x20,0xC6,0xFF,0x20,0xCF,0xFF,0x20,0xCF,0xFF,
  0x20,0xCF,0xFF,0x20,0xCF,0xFF,0x20,0xCF,0xFF,0x85,0x63,0x20,0xCF,0xFF,0x85,0x62,
  0x20,0x4C,0x80,0x20,0xCF,0xFF,0xA6,0x90,0xF0,0x0F,0x20,0xCC,0xFF,0xA9,0x01,0x20,
  0xC3,0xFF,0x20,0xCC,0xFF,0x58,0x4C,0x57,0x82,0x20,0xD2,0xFF,0xC9,0x22,0xD0,0xE3,
  0x20,0xCF,0xFF,0x20,0xD2,0xFF,0xC9,0x00,0xD0,0xF6,0xA9,0x0D,0x20,0xD2,0xFF,0x4C,
  0xD0,0x9B,0x20,0x2F,0x82,0x8E,0xB9,0xC5,0x8E,0xBA,0xC5,0x0E,0xB9,0xC5,0x20,0x31,
  0x80,0xA5,0x7A,0xD0,0x02,0xC6,0x7B,0xC6,0x7A,0x20,0x7B,0x88,0xA5,0xF7,0xD0,0x09,
  0x20,0x14,0x96,0x20,0x21,0x96,0x4C,0x59,0x9C,0xC9,0x01,0xD0,0x09,0x20,0xFA,0x95,
  0x20,0x21,0x96,0x4C,0x59,0x9C,0xC9,0x02,0xD0,0x09,0x20,0x14,0x96,0x20,0x07,0x96,
  0x4C,0x59,0x9C,0x20,0xFA,0x95,0x20,0x07,0x96,0x20,0x29,0x82,0x8E,0xB3,0xC5,0xA9,
  0xF0,0x8D,0xB2,0xC5,0x20,0xE4,0xA2,0xA9,0x00,0x8D,0xB2,0xC5,0x4C,0x57,0x82,0xA5,
  0x09,0xA2,0x00,0xEC,0xB3,0xC5,0xF0,0x0B,0xE8,0xA0,0x00,0xC0,0x0A,0xF0,0xF4,0xC8,
  0x4C,0x7B,0x9C,0xAC,0xB9,0xC5,0x99,0x00,0xD0,0xA5,0x0A,0xC9,0x01,0xD0,0x15,0xAC,
  0xBA,0xC5,0xAD,0x10,0xD0,0x19,0xC0,0x9D,0x8D,0x10,0xD0,0xA5,0xA4,0xAC,0xB9,0xC5,
  0x99,0x01,0xD0,0x60,0xAC,0xBA,0xC5,0xAD,0x10,0xD0,0x39,0xC8,0x9D,0x8D,0x10,0xD0,
  0x4C,0x9B,0x9C,0xA9,0x0A,0x8D,0xD9,0xC5,0x20,0x2F,0x82,0xE0,0x00,0xD0,0x06,0x8E,
  0xD9,0xC5,0x4C,0x57,0x82,0x8E,0xD5,0xC5,0x20,0x29,0x82,0x8E,0xD7,0xC5,0x20,0x29,
  0x82,0x8E,0xD8,0xC5,0xA9,0x00,0x8D,0xD6,0xC5,0x4C,0x57,0x82,0xEE,0x18,0xC5,0xAD,
  0x18,0xC5,0xC9,0x3C,0xD0,0x08,0xEE,0x19,0xC5,0xA9,0x00,0x8D,0x18,0xC5,0xAD,0xD9,
  0xC5,0xC9,0x0A,0xD0,0x2A,0xEE,0xD6,0xC5,0xAD,0xD6,0xC5,0xCD,0xD5,0xC5,0xD0,0x1F,
  0xA9,0x00,0x8D,0xD6,0xC5,0x38,0xAD,0x20,0xD0,0xE9,0xF0,0xCD,0xD7,0xC5,0xF0,0x09,
  0xAD,0xD7,0xC5,0x8D,0x20,0xD0,0x4C,0x1F,0x9D,0xAD,0xD8,0xC5,0x8D,0x20,0xD0,0x4C,
  0x1A,0x9E,0x20,0x2F,0x82,0x86,0xAA,0xAD,0x15,0xD0,0x1D,0xC0,0x9D,0x8D,0x15,0xD0,
  0x20,0x29,0x82,0xA4,0xAA,0xAD,0x88,0x02,0xC9,0xCC,0xF0,0x0E,0xAD,0xB1,0xC5,0xC9,
  0x0A,0xD0,0x0E,0x8A,0x99,0xF8,0xC3,0x4C,0x55,0x9D,0x8A,0x99,0xF8,0xCF,0x4C,0x55,
  0x9D,0x8A,0x99,0xF8,0x07,0x20,0x29,0x82,0xA4,0xAA,0x8A,0x99,0x27,0xD0,0x20,0x29,
  0x82,0xE0,0x01,0xD0,0x0E,0xAD,0x1B,0xD0,0xA6,0xAA,0x1D,0xC0,0x9D,0x8D,0x1B,0xD0,
  0x4C,0x7E,0x9D,0xAD,0x1B,0xD0,0xA6,0xAA,0x3D,0xC8,0x9D,0x8D,0x1B,0xD0,0x20,0x29,
  0x82,0xE0,0x01,0xD0,0x0E,0xAD,0x1C,0xD0,0xA6,0xAA,0x1D,0xC0,0x9D,0x8D,0x1C,0xD0,
  0x4C,0x57,0x82,0xA6,0xAA,0xAD,0x1C,0xD0,0x3D,0xC8,0x9D,0x8D,0x1C,0xD0,0x4C,0x57,
  0x82,0x20,0x2F,0x82,0x8E,0x94,0xCB,0x20,0x31,0x80,0x20,0x58,0x90,0x8E,0x99,0xCB,
  0x8D,0x9A,0xCB,0xA0,0x00,0x8C,0x9B,0xCB,0xA5,0x69,0x8D,0x9C,0xCB,0x4C,0x57,0x82,
  0x01,0x02,0x04,0x08,0x10,0x20,0x40,0x80,0xFE,0xFD,0xFB,0xF7,0xEF,0xDF,0xBF,0x7F,
  0x80,0x40,0x20,0x10,0x08,0x04,0x02,0x01,0xA9,0x0A,0x8D,0xC4,0xC5,0x20,0x2F,0x82,
  0xE0,0x10,0xB0,0x18,0xA9,0x0A,0x9D,0x06,0xC5,0xA0,0x00,0xB1,0x7A,0xC9,0x2C,0xD0,
  0x0B,0x20,0x2F,0x82,0x8E,0xC2,0xC5,0xA9,0x00,0x8D,0xC3,0xC5,0x4C,0x57,0x82,0x20,
  0x2F,0x82,0x8A,0x48,0x20,0x29,0x82,0x18,0x68,0xA8,0x20,0xF0,0xFF,0x20,0x7F,0x80,
  0xA9,0x00,0x85,0x0D,0x20,0x79,0x00,0x4C,0x4D,0x96,0xA5,0xFF,0x48,0xA5,0xFE,0x48,
  0xA5,0xFD,0x48,0xA5,0xFC,0x48,0xAD,0xC4,0xC5,0xC9,0x0A,0xF0,0x03,0x4C,0x98,0x9E,
  0xEE,0xC3,0xC5,0xAD,0xC3,0xC5,0xCD,0xC2,0xC5,0xF0,0x03,0x4C,0x98,0x9E,0xA9,0xD8,
  0x85,0xFF,0xA9,0x00,0x85,0xFE,0xAD,0x88,0x02,0x85,0xFD,0xA9,0x00,0x8D,0xC3,0xC5,
  0x85,0xFC,0xA8,0xB1,0xFE,0x29,0x0F,0xAA,0xBD,0x06,0xC5,0xF0,0x06,0xB1,0xFC,0x49,
  0x80,0x91,0xFC,0xE6,0xFE,0xD0,0x02,0xE6,0xFF,0xE6,0xFC,0xD0,0x02,0xE6,0xFD,0xA5,
  0xFF,0xC9,0xDB,0xD0,0xDE,0xA5,0xFE,0xC9,0xE8,0xD0,0xD8,0x4C,0x98,0x9E,0x20,0x2F,
  0x82,0xE0,0x00,0xD0,0x08,0xA9,0x01,0x8D,0x8E,0xC5,0x4C,0x57,0x82,0xA9,0x0A,0x8D,
  0x8E,0xC5,0x8E,0x8F,0xC5,0x4C,0x57,0x82,0xA5,0xD4,0xF0,0x03,0x4C,0xDE,0x9E,0xAC,
  0x8D,0x02,0xB9,0xDA,0x9E,0x8D,0xD4,0xC5,0xA5,0xCB,0xC9,0x04,0xF0,0x26,0xC9,0x05,
  0xF0,0x1C,0xC9,0x06,0xF0,0x12,0xC9,0x03,0xF0,0x08,0xA9,0x00,0x8D,0xD4,0xC5,0x4C,
  0xDE,0x9E,0xEE,0xD4,0xC5,0xEE,0xD4,0xC5,0xEE,0xD4,0xC5,0xEE,0xD4,0xC5,0xEE,0xD4,
  0xC5,0xEE,0xD4,0xC5,0xEE,0xD4,0xC5,0x4C,0xDE,0x9E,0x00,0x01,0x08,0x09,0xA5,0xCC,
  0xF0,0x03,0x4C,0x47,0x9F,0xAD,0x44,0xC6,0xC9,0x0A,0xF0,0xF6,0xA5,0xD4,0xD0,0xF2,
  0xA5,0xCB,0xCD,0x40,0xC6,0xF0,0x48,0xA5,0xD8,0xD0,0x44,0xAD,0xD4,0xC5,0xF0,0x3F,
  0xAD,0xD4,0xC5,0x85,0xFE,0xC9,0x11,0xB0,0x36,0xA9,0x00,0x85,0xFF,0xC6,0xFE,0x06,
  0xFE,0x26,0xFF,0x06,0xFE,0x26,0xFF,0x06,0xFE,0x26,0xFF,0x06,0xFE,0x26,0xFF,0x18,
  0xA5,0xFE,0x69,0x4B,0x85,0xFE,0xA5,0xFF,0x69,0xC6,0x85,0xFF,0xA0,0x00,0xB1,0xFE,
  0xF0,0x0B,0xC0,0x0F,0xF0,0x07,0x99,0x77,0x02,0xC8,0x4C,0x2E,0x9F,0x84,0xC6,0xA5,
  0xCB,0x8D,0x40,0xC6,0x4C,0x47,0x9F,0xAD,0x8F,0xCB,0xC9,0x0A,0xD0,0x03,0x20,0x1A,
  0x94,0x68,0x85,0xFC,0x68,0x85,0xFD,0x68,0x85,0xFE,0x68,0x85,0xFF,0x4C,0x31,0xEA,
  0x20,0x2F,0x82,0xE0,0x02,0xD0,0x0D,0xA9,0x00,0x8D,0x88,0xCB,0xA9,0x0A,0x8D,0x8F,
  0xCB,0x4C,0x57,0x82,0xE0,0x01,0xD0,0x11,0xA9,0x00,0x8D,0x88,0xCB,0xA9,0x0A,0x8D,
  0x8F,0xCB,0xAD,0x8F,0xCB,0xC9,0x0A,0xF0,0xF9,0x20,0x97,0x94,0x4C,0x57,0x82,0x20,
  0x4E,0x90,0x20,0x58,0x90,0x86,0xAC,0x85,0xAD,0x18,0xA9,0x28,0xE5,0x69,0x85,0x6A,
  0x46,0x6A,0xA9,0x1D,0x20,0xD2,0xFF,0xC6,0x6A,0xA5,0x6A,0xC9,0xFF,0xD0,0xF3,0xA0,
  0x00,0x4C,0xBA,0x9F,0xB1,0xAC,0x20,0xD2,0xFF,0xC8,0xC4,0x69,0xD0,0xF6,0x4C,0x57,
  0x82,0x20,0x2F,0x82,0x20,0xB7,0xA0,0xCA,0x86,0xA6,0x20,0x31,0x80,0xA5,0x7A,0xD0,
  0x02,0xC6,0x7B,0xC6,0x7A,0x20,0x14,0xA0,0xA5,0xAA,0x85,0xA8,0x20,0x31,0x80,0xA5,
  0x7A,0xD0,0x02,0xC6,0x7B,0xC6,0x7A,0x20,0x14,0xA0,0x06,0xA6,0xA6,0xA6,0xBD,0x9C,
  0xA0,0x85,0x20,0xE8,0xBD,0x9C,0xA0,0x85,0x21,0x46,0xA6,0xA0,0x06,0xA5,0xAA,0x91
};

const byte SimonsBasic::abROMA000[8192] = {
  0x20,0xA4,0xA6,0x99,0x9E,0xCB,0xA0,0x05,0xA5,0xA8,0x91,0x20,0xA4,0xA6,0x99,0xA2,
  0xCB,0x4C,0x57,0x82,0x20,0x2F,0x82,0x20,0xB2,0xA0,0x86,0xAA,0x06,0xAA,0x06,0xAA,
  0x06,0xAA,0x06,0xAA,0x20,0x2F,0x82,0x20,0xB2,0xA0,0x8A,0x05,0xAA,0x85,0xAA,0x60,
  0x20,0x4E,0x90,0x20,0x0D,0x80,0x20,0x16,0x80,0x20,0x0F,0x81,0x4C,0x57,0x82,0x20,
  0x2F,0x82,0x20,0xB7,0xA0,0xCA,0x86,0xAA,0x20,0x31,0x80,0x20,0x51,0xA0,0x4C,0x7D,
  0xA0,0xA9,0x00,0x85,0xA6,0x85,0xA8,0xA0,0x00,0xB1,0x7A,0xC9,0x30,0xF0,0x12,0xC9,
  0x31,0xF0,0x05,0xA9,0x04,0x4C,0x4A,0x92,0xA4,0xA6,0xA5,0xA8,0x19,0xD0,0x9D,0x85,
  0xA8,0xE6,0xA6,0x20,0x4E,0x90,0xA5,0xA6,0xC9,0x08,0xD0,0xDB,0x60,0x06,0xAA,0xA4,
  0xAA,0xB9,0x9C,0xA0,0x85,0x20,0xC8,0xB9,0x9C,0xA0,0x85,0x21,0xA0,0x04,0xA5,0xA8,
  0x91,0x20,0x46,0xAA,0xA4,0xAA,0x99,0x48,0xC6,0x4C,0x57,0x82,0x00,0xD4,0x07,0xD4,
  0x0E,0xD4,0xA0,0x00,0xB9,0xE2,0xA7,0xF0,0x06,0x20,0xD2,0xFF,0xC8,0xD0,0xF5,0x4C,
  0x03,0x86,0xE0,0x10,0xB0,0x0A,0x60,0xE0,0x04,0xB0,0x05,0xE0,0x00,0xF0,0x01,0x60,
  0xA9,0x00,0x4C,0x4A,0x92,0xA5,0x2D,0x85,0x20,0xA5,0x2E,0x85,0x21,0xA9,0x0D,0x20,
  0xD2,0xFF,0xA5,0x21,0xA0,0x00,0xC5,0x30,0x90,0x0D,0xF0,0x02,0xB0,0x06,0xA5,0x20,
  0xC5,0x2F,0x90,0x03,0x4C,0x03,0x86,0xB1,0x20,0x85,0xA6,0x09,0x80,0xC5,0xA6,0xD0,
  0x40,0x29,0x7F,0x20,0xD2,0xFF,0x20,0xBF,0xA1,0xB1,0x20,0x29,0x7F,0x20,0xD2,0xFF,
  0xA9,0x25,0x20,0xD2,0xFF,0x20,0x46,0xBF,0x20,0xBF,0xA1,0xB1,0x20,0x85,0x62,0x20,
  0xBF,0xA1,0xB1,0x20,0x85,0x63,0xA2,0x90,0x38,0x20,0x88,0x80,0x20,0xB5,0x80,0x20,
  0xB1,0xA1,0x20,0xBF,0xA1,0x20,0xBF,0xA1,0x20,0xBF,0xA1,0x20,0xBF,0xA1,0x4C,0xCD,
  0xA0,0x29,0x7F,0x20,0xD2,0xFF,0x20,0xBF,0xA1,0xB1,0x20,0x85,0xA6,0x09,0x80,0xC5,
  0xA6,0xD0,0x44,0x29,0x7F,0x20,0xD2,0xFF,0xA9,0x24,0x20,0xD2,0xFF,0x20,0x71,0xBF,
  0xA9,0x22,0x20,0xD2,0xFF,0x20,0xBF,0xA1,0xB1,0x20,0x85,0xA8,0x20,0xBF,0xA1,0xB1,
  0x20,0x85,0x23,0x20,0xBF,0xA1,0xB1,0x20,0x85,0x24,0xA0,0x00,0xB1,0x23,0x20,0xD2,
  0xFF,0xC8,0xC4,0xA8,0xD0,0xF6,0xA9,0x22,0x20,0xD2,0xFF,0x20,0xBF,0xA1,0x20,0xBF,
  0xA1,0x20,0xBF,0xA1,0x4C,0xCD,0xA0,0x29,0x7F,0x20,0xD2,0xFF,0x20,0xBF,0xA1,0xA5,
  0x20,0xA4,0x21,0x20,0xA3,0x80,0x20,0xC6,0xA1,0x20,0xBF,0xA1,0x20,0xBF,0xA1,0x20,
  0xBF,0xA1,0x20,0xB5,0x80,0x20,0xB1,0xA1,0x20,0xBF,0xA1,0x20,0xBF,0xA1,0x4C,0xCD,
  0xA0,0xA2,0x00,0xBD,0x00,0x01,0xF0,0x06,0x20,0xD2,0xFF,0xE8,0xD0,0xF5,0x60,0xE6,
  0x20,0xD0,0x02,0xE6,0x21,0x60,0xA9,0x3D,0x4C,0xFF,0xBE,0xA2,0x00,0x20,0x4E,0x90,
  0x8A,0x48,0x20,0xBE,0x80,0x68,0xAA,0xA5,0x5F,0x9D,0x51,0xCA,0xE8,0xA5,0x60,0x9D,
  0x51,0xCA,0xE8,0xA0,0x00,0xB1,0x5F,0x9D,0x51,0xCA,0xE8,0xC8,0xC0,0x02,0xD0,0xF5,
  0xA0,0x00,0xA9,0xFF,0x91,0x5F,0xB1,0x7A,0xF0,0x0D,0x20,0x4E,0x90,0xB1,0x7A,0xC9,
  0x2C,0xD0,0xCD,0xC9,0x20,0xF0,0xF3,0xA9,0x00,0x9D,0x51,0xCA,0xE8,0x9D,0x51,0xCA,
  0x4C,0x57,0x82,0xA2,0x00,0xBD,0x51,0xCA,0x85,0x5F,0xE8,0xBD,0x51,0xCA,0x85,0x60,
  0xD0,0x03,0x4C,0x03,0x86,0xE8,0xA0,0x00,0xBD,0x51,0xCA,0x91,0x5F,0xE8,0xC8,0xC0,
  0x02,0xD0,0xF5,0xF0,0xE0,0x20,0x4E,0x90,0xA9,0x00,0x8D,0x7E,0xC5,0x8D,0x31,0xC5,
  0x20,0xF0,0x88,0x8C,0x49,0xCB,0x8D,0x4A,0xCB,0x20,0x29,0x82,0x8E,0x47,0xCB,0xA9,
  0x00,0x85,0xA8,0x85,0xA9,0x8D,0x4C,0xCB,0xA9,0x68,0x85,0xAC,0xA9,0x01,0x85,0xAD,
  0xA9,0x0C,0x8D,0x4B,0xCB,0x20,0x29,0x82,0x8E,0x46,0xCB,0x20,0x29,0x82,0x8E,0xC3,
  0xCB,0x20,0x9C,0x88,0x20,0xCA,0xAA,0x4C,0x57,0x82,0x18,0xA5,0x20,0x69,0x28,0x85,
  0x20,0xA5,0x21,0x69,0x00,0x85,0x21,0x60,0x18,0xA5,0x23,0x69,0x28,0x85,0x23,0xA5,
  0x24,0x69,0x00,0x85,0x24,0x60,0x18,0xAD,0x99,0xC5,0xED,0x2F,0xC5,0xAD,0x9A,0xC5,
  0xED,0x30,0xC5,0x90,0x0C,0xAD,0x2F,0xC5,0x8D,0x99,0xC5,0xAD,0x30,0xC5,0x8D,0x9A,
  0xC5,0x18,0xAD,0xA8,0xC5,0xED,0x2F,0xC5,0xAD,0xA9,0xC5,0xED,0x30,0xC5,0x90,0x0C,
  0xAD,0x2F,0xC5,0x8D,0xA8,0xC5,0xAD,0x30,0xC5,0x8D,0xA9,0xC5,0xAD,0xAA,0xC5,0xC9,
  0xC7,0x90,0x05,0xA9,0xC7,0x8D,0xAA,0xC5,0xAD,0xA6,0xC5,0xC9,0xC7,0x90,0x05,0xA9,
  0xC7,0x8D,0xA6,0xC5,0xA9,0x00,0x85,0x9B,0x8D,0xAB,0xC5,0x8D,0xA7,0xC5,0x85,0x9C,
  0x8D,0xA1,0xC5,0x8D,0xA2,0xC5,0xAD,0x99,0xC5,0x85,0x09,0xAD,0x9A,0xC5,0x85,0x0A,
  0xAD,0xAA,0xC5,0x85,0xA4,0xAD,0xB2,0xC5,0xC9,0xF0,0xD0,0x06,0x20,0x6F,0x9C,0x4C,
  0x15,0xA3,0x20,0x32,0x87,0xAD,0xA1,0xC5,0x0D,0xA2,0xC5,0xF0,0x06,0x20,0xFF,0xA3,
  0x4C,0xF6,0xA2,0xAD,0x9A,0xC5,0xCD,0xA9,0xC5,0xD0,0x1F,0xAD,0x99,0xC5,0xCD,0xA8,
  0xC5,0xD0,0x17,0xAD,0xAB,0xC5,0xCD,0xA7,0xC5,0xD0,0x09,0xAD,0xAA,0xC5,0xCD,0xA6,
  0xC5,0xD0,0x01,0x60,0x20,0xFF,0xA3,0x4C,0xF6,0xA2,0x38,0xAD,0xA6,0xC5,0xED,0xAA,
  0xC5,0xA8,0xAD,0xA7,0xC5,0xED,0xAB,0xC5,0xAA,0xB0,0x0F,0x38,0xAD,0xAA,0xC5,0xED,
  0xA6,0xC5,0xA8,0xAD,0xAB,0xC5,0xED,0xA7,0xC5,0xAA,0x18,0x98,0x65,0x9B,0x85,0x9B,
  0x8A,0x65,0x9C,0x85,0x9C,0x38,0xAD,0xA8,0xC5,0xED,0x99,0xC5,0x85,0x12,0xAD,0xA9,
  0xC5,0xED,0x9A,0xC5,0x85,0x13,0xB0,0x11,0x38,0xAD,0x99,0xC5,0xED,0xA8,0xC5,0x85,
  0x12,0xAD,0x9A,0xC5,0xED,0xA9,0xC5,0x85,0x13,0xA9,0x00,0x8D,0xA2,0xC5,0x8D,0xA1,
  0xC5,0x38,0xA5,0x9B,0xE5,0x12,0xA8,0xA5,0x9C,0xE5,0x13,0xAA,0x90,0x0E,0x84,0x9B,
  0x86,0x9C,0xEE,0xA1,0xC5,0xD0,0xEA,0xEE,0xA2,0xC5,0xD0,0xE5,0xAD,0x9A,0xC5,0x8D,
  0x26,0xC5,0xAD,0x99,0xC5,0x8D,0x25,0xC5,0xAD,0xA9,0xC5,0x8D,0x1D,0xC5,0xAD,0xA8,
  0xC5,0x8D,0x1C,0xC5,0x20,0x09,0xA5,0xF0,0x20,0x90,0x0E,0xAD,0x99,0xC5,0xD0,0x03,
  0xCE,0x9A,0xC5,0xCE,0x99,0xC5,0x4C,0xF1,0xA3,0xEE,0x99,0xC5,0xD0,0x03,0xEE,0x9A,
  0xC5,0xAD,0xA1,0xC5,0x0D,0xA2,0xC5,0xD0,0x03,0x4C,0xF6,0xA2,0x4C,0x1D,0xA3,0xAD,
  0xA1,0xC5,0x0D,0xA2,0xC5,0xF0,0x0B,0xAD,0xA1,0xC5,0xD0,0x03,0xCE,0xA2,0xC5,0xCE,
  0xA1,0xC5,0xAD,0xAB,0xC5,0x8D,0x26,0xC5,0xAD,0xAA,0xC5,0x8D,0x25,0xC5,0xAD,0xA7,
  0xC5,0x8D,0x1D,0xC5,0xAD,0xA6,0xC5,0x8D,0x1C,0xC5,0x20,0x09,0xA5,0xF0,0x16,0xB0,
  0x09,0xEE,0xAA,0xC5,0xD0,0x03,0xEE,0xAB,0xC5,0x60,0xAD,0xAA,0xC5,0xD0,0x03,0xCE,
  0xAB,0xC5,0xCE,0xAA,0xC5,0x60,0xA5,0xA6,0xC5,0xA8,0xD0,0x06,0xA5,0xA7,0xC5,0xA9,
  0xF0,0x2A,0xA5,0xA6,0x85,0xA8,0x85,0x63,0xA5,0xA7,0x85,0x62,0x85,0xA9,0xA2,0x90,
  0x38,0x20,0x88,0x80,0x20,0xB5,0x80,0x20,0xB1,0xA1,0x38,0x20,0xF0,0xFF,0x98,0xE9,
  0x0A,0xB0,0xFC,0x49,0xFF,0x69,0x01,0xAA,0xE8,0xCA,0xD0,0x03,0x4C,0xAE,0xA4,0xA9,
  0x20,0x20,0xD2,0xFF,0x4C,0x79,0xA4,0x20,0x4E,0x90,0xA9,0xFF,0x85,0xA8,0x85,0xA9,
  0xA0,0x00,0xA5,0x2B,0x85,0x20,0xA5,0x2C,0x85,0x21,0x4C,0xD1,0xA4,0xB1,0x7A,0xF0,
  0xA5,0xB1,0x20,0xF0,0x09,0xB1,0x7A,0xD1,0x20,0xD0,0x03,0xC8,0xD0,0xEF,0xB1,0x20,
  0xC9,0x22,0xD0,0x03,0x4C,0xF6,0xA4,0x20,0xEE,0x8D,0xA0,0x00,0xB1,0x20,0xD0,0xDD,
  0x20,0xEE,0x8D,0xA0,0x00,0x8C,0xDB,0xC5,0xB1,0x20,0xD0,0x05,0xA9,0x01,0x8D,0xDB,
  0xC5,0x20,0xEE,0x8D,0xB1,0x20,0xD0,0x0A,0xAD,0xDB,0xC5,0xC9,0x01,0xD0,0x03,0x4C,
  0x86,0xE3,0x20,0xEE,0x8D,0xB1,0x20,0x85,0xA6,0x20,0xEE,0x8D,0xB1,0x20,0x85,0xA7,
  0x20,0xEE,0x8D,0x4C,0x9D,0xA4,0x20,0xEE,0x8D,0xB1,0x20,0xC9,0x22,0xF0,0x07,0xC9,
  0x00,0xD0,0xF3,0x4C,0xBA,0xA4,0x4C,0xB7,0xA4,0xAD,0x25,0xC5,0xCD,0x1C,0xC5,0xF0,
  0x09,0xAD,0x26,0xC5,0xED,0x1D,0xC5,0x09,0x01,0x60,0xAD,0x26,0xC5,0xED,0x1D,0xC5,
  0x60,0x48,0x49,0x52,0x45,0x53,0x40,0x50,0x4C,0x4F,0x54,0x40,0x4C,0x49,0x4E,0x45,
  0x40,0x42,0x4C,0x4F,0x43,0x4B,0x40,0x46,0x43,0x48,0x52,0x40,0x46,0x43,0x4F,0x4C,
  0x40,0x46,0x49,0x4C,0x4C,0x40,0x52,0x45,0x43,0x40,0x52,0x4F,0x54,0x40,0x44,0x52,
  0x41,0x57,0x40,0x43,0x48,0x41,0x52,0x40,0x48,0x49,0x20,0x43,0x4F,0x4C,0x40,0x49,
  0x4E,0x56,0x40,0x46,0x52,0x41,0x43,0x40,0x4D,0x4F,0x56,0x45,0x40,0x50,0x4C,0x41,
  0x43,0x45,0x40,0x55,0x50,0x42,0x40,0x55,0x50,0x57,0x40,0x4C,0x45,0x46,0x54,0x57,
  0x40,0x4C,0x45,0x46,0x54,0x42,0x40,0x44,0x4F,0x57,0x4E,0x42,0x40,0x44,0x4F,0x57,
  0x4E,0x57,0x40,0x52,0x49,0x47,0x48,0x54,0x42,0x40,0x52,0x49,0x47,0x48,0x54,0x57,
  0x40,0x4D,0x55,0x4C,0x54,0x49,0x40,0x43,0x4F,0x4C,0x4F,0x55,0x52,0x40,0x4D,0x4D,
  0x4F,0x42,0x40,0x42,0x46,0x4C,0x41,0x53,0x48,0x40,0x4D,0x4F,0x42,0x20,0x53,0x45,
  0x54,0x40,0x4D,0x55,0x53,0x49,0x43,0x40,0x46,0x4C,0x41,0x53,0x48,0x40,0x52,0x45,
  0x50,0x45,0x41,0x54,0x40,0x50,0x4C,0x41,0x59,0x40,0x3E,0x3E,0x40,0x43,0x45,0x4E,
  0x54,0x52,0x45,0x40,0x45,0x4E,0x56,0x45,0x4C,0x4F,0x50,0x45,0x40,0x43,0x47,0x4F,
  0x54,0x4F,0x40,0x57,0x41,0x56,0x45,0x40,0x46,0x45,0x54,0x43,0x48,0x40,0x41,0x54,
  0x28,0x40,0x55,0x4E,0x54,0x49,0x4C,0x40,0x3E,0x3E,0x40,0x3E,0x3E,0x40,0x55,0x53,
  0x45,0x40,0x3E,0x3E,0x40,0x47,0x4C,0x4F,0x42,0x41,0x4C,0x40,0x3E,0x3E,0x40,0x52,
  0x45,0x53,0x45,0x54,0x40,0x50,0x52,0x4F,0x43,0x40,0x43,0x41,0x4C,0x4C,0x40,0x45,
  0x58,0x45,0x43,0x40,0x45,0x4E,0x44,0x20,0x50,0x52,0x4F,0x43,0x40,0x45,0x58,0x49,
  0x54,0x40,0x45,0x4E,0x44,0x20,0x4C,0x4F,0x4F,0x50,0x40,0x4F,0x4E,0x20,0x4B,0x45,
  0x59,0x40,0x44,0x49,0x53,0x41,0x42,0x4C,0x45,0x40,0x52,0x45,0x53,0x55,0x4D,0x45,
  0x40,0x4C,0x4F,0x4F,0x50,0x40,0x44,0x45,0x4C,0x41,0x59,0x40,0x3E,0x3E,0x40,0x3E,
  0x3E,0x40,0x3E,0x3E,0x40,0x3E,0x3E,0x40,0x53,0x45,0x43,0x55,0x52,0x45,0x40,0x44,
  0x49,0x53,0x41,0x50,0x41,0x40,0x43,0x49,0x52,0x43,0x4C,0x45,0x40,0x4F,0x4E,0x20,
  0x45,0x52,0x52,0x4F,0x52,0x40,0x4E,0x4F,0x20,0x45,0x52,0x52,0x4F,0x52,0x40,0x4C,
  0x4F,0x43,0x41,0x4C,0x40,0x52,0x43,0x4F,0x4D,0x50,0x40,0x45,0x4C,0x53,0x45,0x40,
  0x52,0x45,0x54,0x52,0x41,0x43,0x45,0x40,0x54,0x52,0x41,0x43,0x45,0x40,0x44,0x49,
  0x52,0x40,0x50,0x41,0x47,0x45,0x40,0x44,0x55,0x4D,0x50,0x40,0x46,0x49,0x4E,0x44,
  0x40,0x4F,0x50,0x54,0x49,0x4F,0x4E,0x40,0x41,0x55,0x54,0x4F,0x40,0x4F,0x4C,0x44,
  0x40,0x4A,0x4F,0x59,0x40,0x4D,0x4F,0x44,0x40,0x44,0x49,0x56,0x40,0x3E,0x3E,0x40,
  0x44,0x55,0x50,0x40,0x49,0x4E,0x4B,0x45,0x59,0x40,0x49,0x4E,0x53,0x54,0x40,0x54,
  0x45,0x53,0x54,0x40,0x4C,0x49,0x4E,0x40,0x45,0x58,0x4F,0x52,0x40,0x49,0x4E,0x53,
  0x45,0x52,0x54,0x40,0x50,0x4F,0x54,0x40,0x50,0x45,0x4E,0x58,0x40,0x3E,0x3E,0x40,
  0x50,0x45,0x4E,0x59,0x40,0x53,0x4F,0x55,0x4E,0x44,0x40,0x47,0x52,0x41,0x50,0x48,
  0x49,0x43,0x53,0x40,0x44,0x45,0x53,0x49,0x47,0x4E,0x40,0x52,0x4C,0x4F,0x43,0x4D,
  0x4F,0x42,0x40,0x43,0x4D,0x4F,0x42,0x40,0x42,0x43,0x4B,0x47,0x4E,0x44,0x53,0x40,
  0x50,0x41,0x55,0x53,0x45,0x40,0x4E,0x52,0x4D,0x40,0x4D,0x4F,0x42,0x20,0x4F,0x46,
  0x46,0x40,0x4F,0x46,0x46,0x40,0x41,0x4E,0x47,0x4C,0x40,0x41,0x52,0x43,0x40,0x43,
  0x4F,0x4C,0x44,0x40,0x53,0x43,0x52,0x53,0x56,0x40,0x53,0x43,0x52,0x4C,0x44,0x40,
  0x54,0x45,0x58,0x54,0x40,0x43,0x53,0x45,0x54,0x40,0x56,0x4F,0x4C,0x40,0x44,0x49,
  0x53,0x4B,0x40,0x48,0x52,0x44,0x43,0x50,0x59,0x40,0x4B,0x45,0x59,0x40,0x50,0x41,
  0x49,0x4E,0x54,0x40,0x4C,0x4F,0x57,0x20,0x43,0x4F,0x4C,0x40,0x43,0x4F,0x50,0x59,
  0x40,0x4D,0x45,0x52,0x47,0x45,0x40,0x52,0x45,0x4E,0x55,0x4D,0x42,0x45,0x52,0x40,
  0x4D,0x45,0x4D,0x40,0x44,0x45,0x54,0x45,0x43,0x54,0x40,0x43,0x48,0x45,0x43,0x4B,
  0x40,0x44,0x49,0x53,0x50,0x4C,0x41,0x59,0x40,0x45,0x52,0x52,0x40,0x4F,0x55,0x54,
  0x40,0x00,0x44,0x53,0x2B,0x43,0x42,0x4D,0x00,0x20,0x80,0xFF,0x2E,0x30,0x00,0x80,
  0xFF,0x30,0x2E,0x20,0x2F,0x82,0x86,0xAA,0xE0,0x04,0x90,0x05,0xA9,0x0C,0x4C,0x4A,
  0x92,0xA9,0x00,0x8D,0x99,0xC5,0x8D,0xAA,0xC5,0x85,0xA6,0x20,0xED,0x88,0x84,0xAE,
  0x85,0xAF,0xA0,0x00,0x84,0xA8,0xA2,0x00,0xB1,0x7A,0xF0,0x06,0x20,0x4E,0x90,0x4C,
  0x18,0xA8,0x20,0x4E,0x90,0xB1,0x7A,0x8D,0x9A,0xC5,0x20,0x4E,0x90,0xB1,0x7A,0x0D,
  0x9A,0xC5,0xD0,0x05,0xA9,0x0A,0x4C,0x4A,0x92,0x20,0x4E,0x90,0x20,0x4E,0x90,0x20,
  0x4E,0x90,0xA0,0x00,0xB1,0x7A,0xC9,0x40,0xF0,0xF5,0xA5,0xAA,0xF0,0x07,0xC9,0x02,
  0xF0,0x03,0x4C,0xE4,0xA8,0xB1,0x7A,0xC9,0x41,0xF0,0x18,0xC9,0x2E,0xF0,0x14,0xC9,
  0x20,0xF0,0x10,0xC9,0x42,0xF0,0x05,0xA9,0x0B,0x4C,0x4A,0x92,0xA5,0xA8,0x1D,0x4A,
  0x88,0x85,0xA8,0xE8,0xE0,0x08,0xD0,0xC7,0xA4,0xA6,0xA5,0xA8,0x91,0xAE,0xE6,0xA6,
  0xEE,0xAA,0xC5,0xA2,0x00,0x86,0xA8,0xA5,0xAA,0xC9,0x02,0xB0,0x07,0xAD,0xAA,0xC5,
  0xC9,0x03,0xD0,0xAB,0xA0,0x00,0x8C,0xAA,0xC5,0xEE,0x99,0xC5,0xA5,0xAA,0xC9,0x02,
  0x90,0x07,0xAD,0x99,0xC5,0xC9,0x08,0xF0,0x0A,0xAD,0x99,0xC5,0xC9,0x15,0xF0,0x03,
  0x4C,0x12,0xA8,0xB1,0x7A,0xF0,0x06,0x20,0x4E,0x90,0x4C,0xB3,0xA8,0x4C,0x57,0x82,
  0x20,0x4E,0x90,0x20,0x4E,0x90,0x20,0x4E,0x90,0x4C,0x03,0x86,0xC0,0x00,0x30,0x00,
  0x0C,0x00,0x03,0x00,0x80,0x00,0x20,0x00,0x08,0x00,0x02,0x00,0x40,0x00,0x10,0x00,
  0x04,0x00,0x01,0x00,0xA0,0x00,0xB1,0x7A,0xC9,0x44,0xD0,0x0B,0xA5,0xA8,0x1D,0xCC,
  0xA8,0x85,0xA8,0xE8,0x4C,0x73,0xA8,0xC9,0x43,0xD0,0x0B,0xA5,0xA8,0x1D,0xD4,0xA8,
  0x85,0xA8,0xE8,0x4C,0x73,0xA8,0xC9,0x42,0xD0,0x0B,0xA5,0xA8,0x1D,0xDC,0xA8,0x85,
  0xA8,0xE8,0x4C,0x73,0xA8,0xC9,0x2E,0xF0,0x0D,0xC9,0x41,0xF0,0x09,0xC9,0x20,0xF0,
  0x05,0xA9,0x0B,0x4C,0x4A,0x92,0xE8,0x4C,0x73,0xA8,0x20,0x2F,0x82,0x8E,0xB9,0xC5,
  0x8E,0xBA,0xC5,0x0E,0xB9,0xC5,0xAD,0x10,0xD0,0x1D,0xC0,0x9D,0xCD,0x10,0xD0,0xD0,
  0x08,0xA9,0x01,0x8D,0x9A,0xC5,0x4C,0x4E,0xA9,0xA9,0x00,0x8D,0x9A,0xC5,0xAC,0xB9,
  0xC5,0xB9,0x00,0xD0,0x8D,0x99,0xC5,0xB9,0x01,0xD0,0x8D,0xAA,0xC5,0xA9,0x00,0x8D,
  0xAB,0xC5,0x20,0x8D,0x88,0x4C,0x2C,0x9C,0x20,0x2F,0x82,0x8E,0x25,0xD0,0x20,0x29,
  0x82,0x8E,0x26,0xD0,0x4C,0x57,0x82,0xAD,0x11,0xD0,0x29,0xDF,0x09,0x40,0x8D,0x11,
  0xD0,0xAD,0x16,0xD0,0x29,0xEF,0x8D,0x16,0xD0,0x20,0x2F,0x82,0x8E,0x21,0xD0,0x20,
  0x29,0x82,0x8E,0x22,0xD0,0x20,0x29,0x82,0x8E,0x23,0xD0,0x20,0x29,0x82,0x8E,0x24,
  0xD0,0x4C,0x57,0x82,0x20,0x73,0x00,0xA0,0x00,0xB1,0x7A,0xC9,0x22,0xD0,0x23,0xA5,
  0x7A,0xD0,0x02,0xC6,0x7B,0xC6,0x7A,0x20,0x58,0x90,0x85,0xA9,0x86,0xA8,0xA0,0x00,
  0xB1,0xA8,0x20,0xD2,0xFF,0xC8,0xC4,0x69,0xD0,0xF6,0xA9,0x0D,0x20,0xD2,0xFF,0x20,
  0x31,0x80,0x20,0xF0,0x88,0xA9,0x00,0x8D,0x19,0xC5,0x8D,0x18,0xC5,0x20,0xE4,0xFF,
  0xC9,0x0D,0xF0,0x0C,0xAD,0x19,0xC5,0xC5,0x65,0xB0,0x05,0xF0,0x03,0x4C,0xDD,0xA9,
  0x4C,0x57,0x82,0x20,0xF9,0xA9,0x4C,0x03,0x86,0xA9,0x1B,0x8D,0x11,0xD0,0xA9,0x15,
  0x8D,0x18,0xD0,0xA9,0xC8,0x8D,0x16,0xD0,0xA9,0xC7,0x8D,0x00,0xDD,0xA9,0x63,0x8D,
  0xB1,0xC5,0xA9,0x04,0x8D,0x88,0x02,0x60,0x20,0x2F,0x82,0xAD,0x15,0xD0,0x3D,0xC8,
  0x9D,0x8D,0x15,0xD0,0x4C,0x57,0x82,0xA2,0x00,0x8E,0xC4,0xC5,0x8A,0x9D,0x06,0xC5,
  0xE8,0xE0,0x10,0x90,0xF8,0x4C,0x03,0x86,0x20,0x4F,0xAA,0xAD,0x1A,0xC5,0x85,0xA8,
  0xAD,0x1B,0xC5,0x85,0xA9,0xA9,0x0A,0x85,0x21,0x20,0x05,0xAB,0x4C,0x57,0x82,0x20,
  0x4E,0x90,0xA9,0x00,0x8D,0x7E,0xC5,0x8D,0x31,0xC5,0x20,0xF0,0x88,0x8C,0x49,0xCB,
  0x8D,0x4A,0xCB,0x20,0xED,0x88,0x8C,0x47,0xCB,0x20,0xED,0x88,0x84,0xA8,0x85,0xA9,
  0x20,0xED,0x88,0x84,0xAC,0x85,0xAD,0x20,0xED,0x88,0x8D,0x4C,0xCB,0x8C,0x4B,0xCB,
  0xA5,0xA9,0x8D,0x26,0xC5,0xA5,0xA8,0x8D,0x25,0xC5,0xA5,0xAD,0x8D,0x1D,0xC5,0xA5,
  0xAC,0x8D,0x1C,0xC5,0x20,0x09,0xA5,0x90,0x18,0xAD,0x7E,0xC5,0x49,0x0A,0x8D,0x7E,
  0xC5,0xA6,0xAC,0xA4,0xAD,0xA5,0xA8,0x85,0xAC,0xA5,0xA9,0x85,0xAD,0x86,0xA8,0x84,
  0xA9,0xA5,0xAC,0x8D,0x1A,0xC5,0xA5,0xAD,0x8D,0x1B,0xC5,0x20,0x29,0x82,0x8E,0x46,
  0xCB,0x20,0x29,0x82,0x8E,0xC3,0xCB,0x20,0x9C,0x88,0xA9,0x14,0x85,0x21,0xA5,0xA8,
  0x48,0xA5,0xA9,0x48,0x20,0x8A,0xAC,0x20,0xB4,0xAB,0xA5,0x65,0x85,0xAA,0xA5,0x64,
  0x85,0xAB,0x20,0xD2,0xAB,0xA5,0x65,0x85,0xAE,0xA5,0x64,0x85,0xAF,0x20,0x24,0xAC,
  0xA5,0xAE,0x8D,0xA6,0xC5,0xA5,0xAA,0x8D,0xA8,0xC5,0xA5,0xAB,0x8D,0xA9,0xC5,0x68,
  0x85,0xA9,0x68,0x85,0xA8,0xA5,0xA8,0x48,0xA5,0xA9,0x48,0x20,0x8A,0xAC,0x20,0xB4,
  0xAB,0x84,0xAA,0x85,0xAB,0x20,0xD2,0xAB,0x84,0xAE,0x85,0xAF,0x20,0x24,0xAC,0xA5,
  0xAE,0x8D,0xAA,0xC5,0xA5,0xAA,0x8D,0x99,0xC5,0xA5,0xAB,0x8D,0x9A,0xC5,0xAD,0x99,
  0xC5,0x48,0xAD,0xAA,0xC5,0x48,0xAD,0x9A,0xC5,0x48,0x20,0x96,0xA2,0x68,0x8D,0xA9,
  0xC5,0x68,0x8D,0xA6,0xC5,0x68,0x8D,0xA8,0xC5,0x68,0x85,0xA9,0x68,0x85,0xA8,0xA5,
  0x21,0xC9,0x14,0xF0,0x01,0x60,0xAD,0x7E,0xC5,0xC9,0x0A,0xD0,0x21,0x38,0xA5,0xA8,
  0xED,0x4B,0xCB,0x85,0xA8,0xA5,0xA9,0xED,0x4C,0xCB,0x85,0xA9,0xB0,0x1F,0xA9,0x0A,
  0x8D,0x31,0xC5,0xA9,0x68,0x85,0xA8,0xA9,0x01,0x85,0xA9,0x4C,0x8D,0xAB,0x18,0xA5,
  0xA8,0x6D,0x4B,0xCB,0x85,0xA8,0xA5,0xA9,0x6D,0x4C,0xCB,0x85,0xA9,0x8D,0x26,0xC5,
  0xA5,0xA8,0x8D,0x25,0xC5,0xA5,0xAD,0x8D,0x1D,0xC5,0xA5,0xAC,0x8D,0x1C,0xC5,0xAD,
  0x7E,0xC5,0xC9,0x0A,0xD0,0x03,0x4C,0x04,0xAD,0x20,0x09,0xA5,0xF0,0x03,0x90,0x01,
  0x60,0x4C,0x05,0xAB,0x20,0xF0,0xAB,0x20,0x1F,0x80,0x20,0x91,0x80,0xAD,0x46,0xCB,
  0x85,0x63,0xA5,0xA9,0x85,0x62,0xA2,0x90,0x38,0x20,0x88,0x80,0x20,0x9A,0x80,0x4C,
  0x16,0x80,0x20,0xF0,0xAB,0x20,0x28,0x80,0x20,0x91,0x80,0xAD,0xC3,0xCB,0x85,0x63,
  0xA5,0xA9,0x85,0x62,0xA2,0x90,0x38,0x20,0x88,0x80,0x20,0x9A,0x80,0x4C,0x16,0x80,
  0xA5,0xA8,0x85,0x63,0xA5,0xA9,0x85,0x62,0xA2,0x90,0x38,0x20,0x88,0x80,0x20,0x91,
  0x80,0xA9,0xA8,0xA0,0xAE,0x20,0xA3,0x80,0x20,0x9A,0x80,0x20,0x91,0x80,0xA9,0xB4,
  0x85,0x63,0xA9,0x00,0x85,0x62,0xA2,0x90,0x38,0x20,0x88,0x80,0xA5,0x61,0xA4,0x70,
  0x20,0xAC,0x80,0x60,0xA5,0x20,0xC9,0x00,0xD0,0x06,0x20,0x7B,0xAC,0x4C,0x4A,0xAC,
  0xC9,0x01,0xD0,0x06,0x20,0x5A,0xAC,0x4C,0x4A,0xAC,0xC9,0x02,0xD0,0x06,0x20,0x63,
  0xAC,0x4C,0x5A,0xAC,0x20,0x63,0xAC,0x4C,0x7B,0xAC,0x18,0xA5,0xAA,0x6D,0x49,0xCB,
  0x85,0xAA,0xA5,0xAB,0x6D,0x4A,0xCB,0x85,0xAB,0x60,0x18,0xA5,0xAE,0x6D,0x47,0xCB,
  0x85,0xAE,0x60,0x38,0xAD,0x49,0xCB,0xE5,0xAA,0x85,0xAA,0xAD,0x4A,0xCB,0xE5,0xAB,
  0x85,0xAB,0xB0,0x06,0xA9,0x00,0x85,0xAA,0x85,0xAB,0x60,0x38,0xAD,0x47,0xCB,0xE5,
  0xAE,0x85,0xAE,0xB0,0xF5,0xA9,0x00,0x85,0xAE,0x60,0xA9,0x5A,0x8D,0xB9,0xCB,0xA9,
  0x00,0x8D,0xBA,0xCB,0xA5,0xA8,0x8D,0xBE,0xCB,0xA5,0xA9,0x8D,0xBF,0xCB,0x20,0x04,
  0x99,0xAD,0xC0,0xCB,0x85,0xA8,0xAD,0xC1,0xCB,0x85,0xA9,0xAD,0xBE,0xCB,0x85,0x20,
  0xC9,0x01,0xF0,0x05,0xC9,0x03,0xF0,0x01,0x60,0x38,0xA9,0x5A,0xE5,0xA8,0x85,0xA8,
  0xA5,0x20,0x60,0x20,0x4E,0x90,0x20,0xF0,0x88,0x8C,0x49,0xCB,0x8C,0xA8,0xC5,0x8D,
  0x4A,0xCB,0x8D,0xA9,0xC5,0x20,0xED,0x88,0x8C,0x47,0xCB,0x8C,0xA6,0xC5,0x8D,0x48,
  0xCB,0x8D,0xA7,0xC5,0x20,0xED,0x88,0x84,0xA8,0x85,0xA9,0x20,0x29,0x82,0x8E,0x46,
  0xCB,0x20,0x29,0x82,0x8E,0xC3,0xCB,0x20,0x9C,0x88,0xA9,0x0A,0x85,0x21,0x20,0x05,
  0xAB,0x4C,0x57,0x82,0x20,0x09,0xA5,0xF0,0x0A,0xB0,0x08,0xAD,0x31,0xC5,0xC9,0x0A,
  0xD0,0x01,0x60,0x4C,0x05,0xAB,0xC8,0xB9,0xFB,0x01,0xF0,0x18,0xC9,0x22,0xD0,0xF6,
  0x4C,0x2E,0xAD,0xC8,0xB9,0xFB,0x01,0xF0,0x06,0xD0,0xF8,0xA0,0x05,0xA9,0xC8,0xB9,
  0xFB,0x01,0xD0,0x01,0x60,0xC9,0x22,0xF0,0xDD,0xC9,0x40,0xF0,0xE6,0xC9,0x41,0x90,
  0xED,0xC9,0x5B,0xB0,0xE9,0x8C,0x52,0xCB,0xA2,0xA5,0x86,0x21,0xA2,0x21,0x86,0x20,
  0xA2,0x00,0x8E,0xE2,0xC5,0xC1,0x20,0xF0,0x1D,0xA1,0x20,0xC9,0x40,0xF0,0x23,0xA1,
  0x20,0xF0,0xCB,0xC9,0x40,0xF0,0x08,0xE6,0x20,0xD0,0xF4,0xE6,0x21,0xD0,0xF0,0xEE,
  0xE2,0xC5,0xAC,0x52,0xCB,0xA9,0xC8,0xB9,0xFB,0x01,0xE6,0x20,0xD0,0xD7,0xE6,0x21,
  0xD0,0xD3,0xAE,0x52,0xCB,0xA9,0x64,0x9D,0xFB,0x01,0xE8,0xAD,0xE2,0xC5,0xC9,0x40,
  0xD0,0x1E,0x9D,0xFB,0x01,0xFE,0xFB,0x01,0xA9,0x3A,0xE8,0x9D,0xFB,0x01,0xE8,0x9D,
  0xFB,0x01,0xE8,0x9D,0xFB,0x01,0xE8,0x9D,0xFB,0x01,0xE8,0x8A,0xA8,0x4C,0x2F,0xAD,
  0x9D,0xFB,0x01,0xFE,0xFB,0x01,0x88,0xC8,0xE8,0xB9,0xFB,0x01,0x9D,0xFB,0x01,0xD0,
  0xF6,0xE8,0xA9,0x00,0x9D,0xFB,0x01,0xBD,0xFC,0x01,0xD0,0xF5,0xAC,0x52,0xCB,0xC8,
  0x4C,0x2E,0xAD,0xA9,0xFF,0x85,0xA6,0xA5,0x7A,0x85,0xA8,0xA5,0x7B,0x85,0xA9,0xE6,
  0xA6,0xA5,0xA6,0xC9,0x02,0xD0,0x03,0x4C,0x57,0x82,0xA5,0xA8,0x85,0x7A,0xA5,0xA9,
  0x85,0x7B,0x18,0xA0,0x00,0xB1,0x7A,0x69,0x01,0x4C,0x75,0xAE,0x68,0x68,0xA9,0x0C,
  0x4C,0x4A,0x92,0x20,0x2F,0x82,0xE0,0x29,0xB0,0xF2,0x8E,0x74,0xCB,0x20,0x29,0x82,
  0xE0,0x28,0xB0,0xE8,0x8E,0x75,0xCB,0x20,0x29,0x82,0xE0,0x00,0xF0,0xDE,0x8E,0x76,
  0xCB,0x20,0x29,0x82,0xE0,0x00,0xF0,0xD4,0x8E,0x77,0xCB,0xA9,0x00,0x8D,0x78,0xCB,
  0xAD,0x88,0x02,0x85,0x21,0xA5,0xA6,0xF0,0x04,0xA9,0xD8,0x85,0x21,0xAC,0x74,0xCB,
  0x18,0xB9,0x28,0x89,0x85,0xA4,0xB9,0x41,0x89,0x65,0x21,0x85,0xA5,0x18,0xA5,0xA4,
  0x6D,0x75,0xCB,0x85,0x20,0xA5,0xA5,0x69,0x00,0x85,0x21,0x18,0xAD,0x74,0xCB,0x6D,
  0x77,0xCB,0xC9,0x1A,0xB0,0x0C,0x18,0xAD,0x76,0xCB,0x6D,0x75,0xCB,0xC9,0x29,0xB0,
  0x01,0x60,0x4C,0xFC,0xAD,0xC9,0x0E,0xD0,0x26,0x20,0x03,0xAE,0xA0,0x00,0xA2,0x00,
  0xCC,0x76,0xCB,0xF0,0x09,0xB1,0x20,0x49,0x80,0x91,0x20,0xC8,0xD0,0xF2,0xE8,0xEC,
  0x77,0xCB,0xF0,0x08,0xA0,0x00,0x20,0x7A,0xA2,0x4C,0x80,0xAE,0x4C,0x57,0x82,0xC9,
  0x10,0xF0,0x03,0x4C,0xE5,0xAE,0x20,0x03,0xAE,0xA5,0x20,0x85,0x23,0xA5,0x21,0x85,
  0x24,0x20,0x29,0x82,0x8E,0x74,0xCB,0x20,0x29,0x82,0x8E,0x75,0xCB,0x20,0x2B,0xAE,
  0xA2,0x00,0xA0,0x00,0xCC,0x76,0xCB,0xF0,0x08,0xB1,0x23,0x91,0x20,0xC8,0x4C,0xC4,
  0xAE,0xE8,0xEC,0x77,0xCB,0xF0,0x0B,0xA0,0x00,0x20,0x7A,0xA2,0x20,0x88,0xA2,0x4C,
  0xC4,0xAE,0x4C,0xDF,0xAD,0xC9,0x12,0xF0,0x03,0x4C,0x89,0xAF,0xA9,0x28,0x8D,0x7A,
  0xCB,0x20,0xF7,0xAE,0x4C,0x79,0xAF,0x20,0x03,0xAE,0x18,0xA5,0x20,0x6D,0x7A,0xCB,
  0x85,0x23,0xA5,0x21,0x69,0x00,0x85,0x24,0xA2,0x00,0xA0,0x00,0xAD,0x7A,0xCB,0xC9,
  0x28,0xF0,0x30,0xCE,0x76,0xCB,0xAD,0x76,0xCB,0xD0,0x03,0xEE,0x76,0xCB,0xA5,0x20,
  0x8D,0x7C,0xCB,0xA5,0x21,0x8D,0x7E,0xCB,0xB1,0x20,0x9D,0x1C,0xCB,0xE8,0x20,0x7A,
  0xA2,0xEC,0x77,0xCB,0xD0,0xF2,0xAD,0x7E,0xCB,0x85,0x21,0xAD,0x7C,0xCB,0x85,0x20,
  0x4C,0x51,0xAF,0xB1,0x20,0x99,0x1C,0xCB,0xC8,0xCC,0x76,0xCB,0xD0,0xF5,0xCE,0x77,
  0xCB,0xA0,0x00,0xA2,0x00,0xCC,0x76,0xCB,0xF0,0x08,0xB1,0x23,0x91,0x20,0xC8,0x4C,
  0x55,0xAF,0xE8,0xEC,0x77,0xCB,0xF0,0x10,0xAD,0x77,0xCB,0xF0,0x0B,0xA0,0x00,0x20,
  0x7A,0xA2,0x20,0x88,0xA2,0x4C,0x55,0xAF,0x60,0xAC,0x76,0xCB,0x88,0xA9,0x20,0x91,
  0x23,0x88,0xC0,0xFF,0xD0,0xF9,0x4C,0xDF,0xAD,0xC9,0x13,0xF0,0x03,0x4C,0xA9,0xAF,
  0xA9,0x28,0x8D,0x7A,0xCB,0x20,0xF7,0xAE,0xAC,0x76,0xCB,0x88,0xB9,0x1C,0xCB,0x91,
  0x23,0x88,0xC0,0xFF,0xD0,0xF6,0x4C,0xDF,0xAD,0xC9,0x14,0xF0,0x03,0x4C,0x02,0xB0,
  0xA9,0x01,0x8D,0x7A,0xCB,0x8D,0x80,0xCB,0x20,0xF7,0xAE,0xA2,0x00,0xAC,0x77,0xCB,
  0xC0,0x01,0xF0,0x12,0x18,0xA5,0x23,0xE9,0x27,0x85,0x23,0xA5,0x24,0xE9,0x00,0x85,
  0x24,0x88,0xC0,0x01,0xD0,0xEE,0xAC,0x76,0xCB,0x88,0xAD,0x80,0xCB,0xC9,0x04,0xF0,
  0x11,0xBD,0x1C,0xCB,0x91,0x23,0x20,0x88,0xA2,0xE8,0xEC,0x77,0xCB,0xD0,0xF2,0x4C,
  0xDF,0xAD,0xA9,0x20,0x91,0x23,0x20,0x88,0xA2,0xE8,0xEC,0x77,0xCB,0xD0,0xF3,0x4C,
  0xDF,0xAD,0xC9,0x15,0xF0,0x03,0x4C,0x19,0xB0,0xA9,0x01,0x8D,0x7A,0xCB,0xA9,0x04,
  0x8D,0x80,0xCB,0x20,0xF7,0xAE,0x4C,0xBB,0xAF,0xC9,0x16,0xF0,0x03,0x4C,0xE3,0xB0,
  0xA9,0x28,0x8D,0x7A,0xCB,0xA9,0x00,0x8D,0x7C,0xCB,0x20,0x30,0xB0,0x4C,0xD4,0xB0,
  0x20,0x03,0xAE,0xCE,0x77,0xCB,0xCE,0x76,0xCB,0xAE,0x77,0xCB,0xA0,0x00,0x18,0x8C,
  0x82,0xCB,0xAC,0x76,0xCB,0xB1,0x20,0xAC,0x82,0xCB,0x99,0x1C,0xCB,0xAD,0x77,0xCB,
  0xF0,0x0A,0xC8,0x20,0x7A,0xA2,0x8C,0x82,0xCB,0xCA,0xD0,0xE2,0x8C,0x82,0xCB,0xAC,
  0x76,0xCB,0xB1,0x20,0xAC,0x82,0xCB,0x99,0x1C,0xCB,0x38,0xA5,0x20,0xED,0x7A,0xCB,
  0x85,0x23,0xA5,0x21,0xE9,0x00,0x85,0x24,0xAD,0x7C,0xCB,0xC9,0x28,0xD0,0x1D,0xA0,
  0x00,0xEE,0x76,0xCB,0xAD,0x7E,0xCB,0xC9,0x02,0xF0,0x0E,0xB1,0x20,0x99,0x1C,0xCB,
  0xC8,0xCC,0x76,0xCB,0xF0,0x03,0x4C,0x8B,0xB0,0xCE,0x76,0xCB,0xAC,0x76,0xCB,0xAE,
  0x77,0xCB,0xC0,0xFF,0xF0,0x08,0xB1,0x23,0x91,0x20,0x88,0x4C,0xA2,0xB0,0xCA,0xE0,
  0xFF,0xF0,0x20,0xAC,0x76,0xCB,0x38,0xA5,0x20,0xE9,0x28,0x85,0x20,0xA5,0x21,0xE9,
  0x00,0x85,0x21,0x38,0xA5,0x23,0xE9,0x28,0x85,0x23,0xA5,0x24,0xE9,0x00,0x85,0x24,
  0x4C,0xA2,0xB0,0x60,0xAC,0x76,0xCB,0xA9,0x20,0x91,0x20,0x88,0xC0,0xFF,0xD0,0xF9,
  0x4C,0xDF,0xAD,0xC9,0x17,0xF0,0x03,0x4C,0x0E,0xB1,0xA9,0x28,0x8D,0x7A,0xCB,0x8D,
  0x7C,0xCB,0x8D,0x7E,0xCB,0x20,0x30,0xB0,0xEE,0x76,0xCB,0xA0,0x00,0xB9,0x1C,0xCB,
  0x91,0x20,0xC8,0xCC,0x76,0xCB,0x10,0x03,0x4C,0xFD,0xB0,0x4C,0xDF,0xAD,0xC9,0x18,
  0xF0,0x03,0x4C,0x3A,0xB1,0xA9,0x01,0x8D,0x7A,0xCB,0x8D,0x7C,0xCB,0x20,0x30,0xB0,
  0xA2,0x00,0xEE,0x77,0xCB,0xA0,0x00,0xA9,0x20,0x91,0x20,0xE8,0x20,0x7A,0xA2,0xEC,
  0x77,0xCB,0x10,0x03,0x4C,0x27,0xB1,0x4C,0xDF,0xAD,0xC9,0x19,0xF0,0x00,0xA9,0x01,
  0x8D,0x7A,0xCB,0x8D,0x7C,0xCB,0xA9,0x02,0x8D,0x7E,0xCB,0x20,0x30,0xB0,0xA0,0x00,
  0xA2,0x00,0xEE,0x77,0xCB,0xBD,0x1C,0xCB,0x91,0x20,0xE8,0x20,0x7A,0xA2,0xEC,0x77,
  0xCB,0xF0,0x03,0x4C,0x55,0xB1,0xA9,0x01,0x8D,0x82,0xCB,0x4C,0xDF,0xAD,0x20,0x55,
  0x90,0x86,0x20,0x85,0x21,0xA5,0x69,0x8D,0x70,0xCB,0x20,0x29,0x82,0x8E,0x6D,0xCB,
  0x20,0x31,0x80,0x20,0xDA,0xB2,0x20,0xC7,0x80,0xA9,0x2C,0x8D,0xFF,0x01,0x20,0xBD,
  0xB1,0xA5,0x13,0xF0,0x10,0x20,0xB7,0xFF,0x29,0x02,0xF0,0x09,0x20,0xD0,0x80,0x20,
  0xD9,0x80,0x4C,0x57,0x82,0xAD,0x00,0x02,0xD0,0x0D,0xA5,0x13,0xD0,0xE0,0x20,0xE2,
  0x80,0x20,0xFD,0x80,0x4C,0x57,0x82,0x20,0xEB,0x80,0x4C,0x57,0x82,0xA2,0x00,0x98,
  0x48,0x8A,0x48,0x20,0xE4,0xFF,0x8D,0x6E,0xCB,0x68,0xAA,0x68,0xA8,0xAD,0x6E,0xCB,
  0xC9,0x00,0xF0,0xEB,0xC9,0x0D,0xD0,0x03,0x4C,0x77,0xB2,0xC9,0x14,0xF0,0x22,0xA5,
  0xD4,0xD0,0x36,0x8A,0x48,0xA2,0x00,0xBD,0xC5,0xB2,0xCD,0x6E,0xCB,0xF0,0x0D,0xE8,
  0xE0,0x12,0xD0,0xF3,0x68,0xAA,0xAD,0x6E,0xCB,0x4C,0x19,0xB2,0x68,0xAA,0x4C,0xBF,
  0xB1,0xE0,0x00,0xF0,0xBA,0xA5,0xD4,0x48,0xA9,0x00,0x85,0xD4,0xA9,0x14,0x20,0xD2,
  0xFF,0x68,0x85,0xD4,0xCA,0xCA,0x4C,0x59,0xB2,0x8C,0x6F,0xCB,0xAD,0x6E,0xCB,0xC9,
  0x8D,0xF0,0xE0,0xAC,0x71,0xCB,0xC0,0x78,0xF0,0x26,0xA0,0x00,0xD1,0x20,0xF0,0x20,
  0xB1,0x20,0xC9,0x13,0xF0,0x57,0xC9,0x11,0xF0,0x64,0xC9,0x1D,0xD0,0x03,0x4C,0xAC,
  0xB2,0xAD,0x6E,0xCB,0xC8,0xCC,0x70,0xCB,0xD0,0xE2,0xAC,0x6F,0xCB,0x4C,0xBF,0xB1,
  0x9D,0x00,0x02,0xAC,0x6F,0xCB,0x20,0xD2,0xFF,0x20,0xDA,0xB2,0xE8,0xEC,0x6D,0xCB,
  0xD0,0xE8,0x98,0x48,0x8A,0x48,0x20,0xE4,0xFF,0x8D,0x6E,0xCB,0x68,0xAA,0x68,0xA8,
  0xAD,0x6E,0xCB,0xC9,0x0D,0xD0,0xEB,0xE0,0x00,0xD0,0x03,0x4C,0xBD,0xB1,0xA9,0x20,
  0x20,0xD2,0xFF,0xA9,0x91,0x20,0xD2,0xFF,0xA9,0x0D,0x4C,0xF4,0x80,0xAD,0x6E,0xCB,
  0xC9,0x41,0x90,0x07,0xC9,0x5B,0xB0,0x03,0x4C,0x50,0xB2,0x4C,0x41,0xB2,0xAD,0x6E,
  0xCB,0xC9,0x1F,0x90,0xF6,0xC9,0x41,0xB0,0xF2,0x4C,0x50,0xB2,0xAD,0x6E,0xCB,0xC9,
  0x41,0x90,0xE8,0xC9,0x5B,0xB0,0x03,0x4C,0x50,0xB2,0xC9,0xC1,0x90,0xDD,0xC9,0xDB,
  0xB0,0xD9,0x4C,0x50,0xB2,0x05,0x11,0x12,0x13,0x1C,0x1D,0x1E,0x1F,0x90,0x91,0x92,
  0x93,0x94,0x9C,0x9D,0x9E,0x9F,0x00,0x00,0x00,0x00,0xA5,0xD4,0x48,0xA9,0x00,0x85,
  0xD4,0xA9,0x12,0x20,0xD2,0xFF,0xA9,0x20,0x20,0xD2,0xFF,0xA9,0x9D,0x20,0xD2,0xFF,
  0xA9,0x92,0x20,0xD2,0xFF,0x68,0x85,0xD4,0x60,0x20,0x4E,0x90,0x20,0x23,0x81,0xA6,
  0xB8,0x20,0xC9,0xFF,0xA2,0x00,0xBD,0x00,0x04,0x20,0xD2,0xFF,0xBD,0x00,0xD8,0x20,
  0xD2,0xFF,0xBD,0xFA,0x04,0x20,0xD2,0xFF,0xBD,0xFA,0xD8,0x20,0xD2,0xFF,0xBD,0xF4,
  0x05,0x20,0xD2,0xFF,0xBD,0xF4,0xD9,0x20,0xD2,0xFF,0xBD,0xEE,0x06,0x20,0xD2,0xFF,
  0xBD,0xEE,0xDA,0x20,0xD2,0xFF,0xE8,0xE0,0xFA,0xD0,0xCB,0x20,0xCC,0xFF,0xA5,0xB8,
  0x20,0xC3,0xFF,0x4C,0x57,0x82,0x20,0x4E,0x90,0x20,0x23,0x81,0xA6,0xB8,0x20,0xC6,
  0xFF,0xA2,0x00,0x20,0xCF,0xFF,0x9D,0x00,0x04,0x20,0xCF,0xFF,0x9D,0x00,0xD8,0x20,
  0xCF,0xFF,0x9D,0xFA,0x04,0x20,0xCF,0xFF,0x9D,0xFA,0xD8,0x20,0xCF,0xFF,0x9D,0xF4,
  0x05,0x20,0xCF,0xFF,0x9D,0xF4,0xD9,0x20,0xCF,0xFF,0x9D,0xEE,0x06,0x20,0xCF,0xFF,
  0x9D,0xEE,0xDA,0xE8,0xE0,0xFA,0xD0,0xCB,0x20,0xCC,0xFF,0xA5,0xB8,0x20,0xC3,0xFF,
  0x4C,0x57,0x82,0x20,0x4E,0x90,0x20,0xF0,0x88,0x84,0x09,0x85,0x0A,0x20,0x29,0x82,
  0x86,0xA4,0x20,0x31,0x80,0x20,0x58,0x90,0x85,0x21,0x86,0x20,0xA5,0x69,0x85,0x23,
  0x20,0x9C,0x88,0x20,0x29,0x82,0x8E,0x6C,0xCB,0xE0,0x00,0xD0,0x05,0xA9,0x01,0x8D,
  0x6C,0xCB,0x20,0x29,0x82,0x8E,0x46,0xCB,0xA0,0x00,0x8C,0x4B,0xCB,0x84,0x6A,0xA5,
  0xA4,0x48,0xA5,0x09,0x48,0xA5,0x0A,0x48,0xAD,0x6C,0xCB,0x48,0xB1,0x20,0xC9,0x12,
  0xD0,0x07,0xA9,0x80,0x85,0x6A,0x4C,0xB0,0xB4,0xC9,0x92,0xD0,0x07,0xA9,0x00,0x85,
  0x6A,0x4C,0xB0,0xB4,0xC9,0x01,0xD0,0x08,0xA9,0x00,0x8D,0x4B,0xCB,0x4C,0xB0,0xB4,
  0xC9,0x02,0xD0,0x08,0xA9,0x01,0x8D,0x4B,0xCB,0x4C,0xB0,0xB4,0x85,0xAC,0xC9,0xFF,
  0xD0,0x05,0xA9,0x7E,0x4C,0x28,0xB4,0xC9,0xE0,0x90,0x06,0x38,0xE9,0x40,0x4C,0x28,
  0xB4,0xC9,0xC0,0x90,0x03,0x38,0xE9,0x60,0xC9,0x40,0x90,0x15,0xC9,0x60,0xB0,0x05,
  0x29,0xBF,0x4C,0x41,0xB4,0xC9,0x80,0xB0,0x05,0x29,0xDF,0x4C,0x41,0xB4,0x38,0xE9,
  0x40,0x45,0x6A,0x85,0xAC,0xAD,0x4B,0xCB,0x85,0xAD,0x98,0x48,0x20,0xD1,0x8A,0x68,
  0xA8,0x68,0x8D,0x6C,0xCB,0x68,0x85,0x0A,0x68,0x85,0x09,0x68,0x85,0xA4,0x18,0xA5,
  0x09,0x6D,0x46,0xCB,0x85,0x09,0x90,0x02,0xE6,0x0A,0xC8,0xC4,0x23,0xD0,0x03,0x4C,
  0x57,0x82,0x4C,0xCF,0xB3,0x20,0x2F,0x82,0xE0,0x02,0xD0,0x1C,0xA9,0x95,0x8D,0x00,
  0xDD,0xA9,0x3B,0x8D,0x11,0xD0,0xA9,0x0B,0x8D,0x18,0xD0,0xA9,0x94,0x8D,0x00,0xDD,
  0xA9,0x0A,0x8D,0xB1,0xC5,0x4C,0x57,0x82,0x20,0xF9,0xA9,0xE0,0x01,0xD0,0x05,0xA9,
  0x0E,0x20,0xD2,0xFF,0x4C,0x57,0x82,0x20,0x2F,0x82,0x8E,0x18,0xD4,0x4C,0x57,0x82,
  0x68,0x8D,0x6C,0xCB,0x68,0x85,0x0A,0x68,0x85,0x09,0x68,0x85,0xA4,0xC8,0xC4,0x23,
  0xD0,0xB0,0x4C,0x57,0x82,0x20,0x4E,0x90,0x20,0xCC,0xFF,0xA9,0x01,0x20,0xC3,0xFF,
  0x20,0xCC,0xFF,0x20,0x58,0x90,0xA8,0xA5,0x69,0x20,0xBD,0xFF,0xA9,0x01,0xA2,0x08,
  0xA0,0x0F,0x20,0xBA,0xFF,0x20,0xC0,0xFF,0xA2,0x01,0x20,0xC9,0xFF,0x20,0xCC,0xFF,
  0xA9,0x01,0x20,0xC3,0xFF,0x20,0xCC,0xFF,0x4C,0x57,0x82,0x20,0x4E,0x90,0x20,0x58,
  0x90,0x85,0xA7,0x86,0xA6,0xA5,0x69,0x85,0xAE,0xA0,0x00,0xB1,0xA6,0xC9,0x23,0xD0,
  0x02,0xA9,0x20,0x99,0x12,0x02,0xC8,0xC4,0x69,0xD0,0xF0,0x8C,0x0D,0x02,0x20,0x31,
  0x80,0x20,0x58,0x90,0x85,0xA9,0x86,0xA8,0xA5,0x69,0x85,0xAF,0xA0,0x00,0x84,0xA4,
  0x84,0x09,0xA2,0x00,0xB1,0xA6,0xC9,0x2E,0xF0,0x0C,0xE6,0x09,0xE8,0x20,0x5A,0x89,
  0xA5,0x09,0xC5,0xAE,0xD0,0xEE,0xA5,0x09,0x85,0x0A,0x8E,0x77,0xCB,0xB1,0xA8,0xC9,
  0x2E,0xF0,0x0B,0xE6,0xA4,0x20,0x61,0x89,0xA5,0xA4,0xC5,0xAF,0xD0,0xEF,0xA5,0xA4,
  0x85,0xA5,0xA5,0xA9,0x8D,0xAB,0xC5,0xA5,0xA8,0x8D,0xAA,0xC5,0xA5,0xA7,0x8D,0x9A,
  0xC5,0xA5,0xA6,0x8D,0x99,0xC5,0x20,0x61,0x89,0xE6,0xA4,0xA5,0x09,0xC5,0xAE,0xB0,
  0x1E,0xA5,0xA4,0xC5,0xAF,0xB0,0x18,0xB1,0xA6,0xC9,0x23,0xD0,0x07,0xB1,0xA8,0x20,
  0x61,0x89,0xE6,0xA4,0x20,0x03,0xB6,0xE6,0x09,0x20,0x5A,0x89,0x4C,0x7B,0xB5,0xAD,
  0x99,0xC5,0x85,0xA6,0xAD,0x9A,0xC5,0x85,0xA7,0xAD,0xAA,0xC5,0x85,0xA8,0xAD,0xAB,
  0xC5,0x85,0xA9,0xAE,0x77,0xCB,0xA5,0xA8,0xD0,0x02,0xC6,0xA9,0xC6,0xA8,0xC6,0xA5,
  0xA5,0x0A,0xC9,0xFF,0xF0,0x2C,0xA5,0xA5,0xC9,0xFF,0xF0,0x26,0xB1,0xA6,0xC9,0x23,
  0xD0,0x0E,0xB1,0xA8,0x48,0xC6,0xA5,0xA5,0xA8,0xD0,0x02,0xC6,0xA9,0xC6,0xA8,0x68,
  0x20,0x03,0xB6,0xCA,0xCA,0xC6,0x0A,0xA5,0xA6,0xD0,0x02,0xC6,0xA7,0xC6,0xA6,0x4C,
  0xC0,0xB5,0xA2,0x00,0xBD,0x12,0x02,0x20,0xD2,0xFF,0xE8,0xEC,0x0D,0x02,0xD0,0xF4,
  0x4C,0x57,0x82,0x9D,0x12,0x02,0xE8,0x60,0x20,0x4E,0x90,0x20,0xCC,0xFF,0xA9,0x01,
  0x20,0xC3,0xFF,0x20,0xCC,0xFF,0xA9,0x01,0xA2,0x04,0xA0,0xFF,0x20,0xBA,0xFF,0x20,
  0xC0,0xFF,0xA2,0x01,0x20,0xC9,0xFF,0xA9,0x0D,0x20,0xD2,0xFF,0xA9,0x04,0x85,0x21,
  0xA9,0x00,0x85,0x20,0xA0,0x00,0xA2,0x00,0xB1,0x20,0xC9,0x80,0x90,0x09,0x49,0x80,
  0x48,0xA9,0x12,0x20,0xD2,0xFF,0x68,0xC9,0x20,0xB0,0x13,0x48,0xAD,0x18,0xD0,0xC9,
  0x17,0xD0,0x05,0xA9,0x11,0x20,0xD2,0xFF,0x68,0x09,0x40,0x4C,0x80,0xB6,0xC9,0x40,
  0x90,0x1D,0xC9,0x7F,0xF0,0x14,0x48,0xAD,0x18,0xD0,0xC9,0x17,0xD0,0x05,0xA9,0x11,
  0x20,0xD2,0xFF,0x68,0x18,0x69,0x80,0x4C,0x80,0xB6,0xA9,0xBF,0x4C,0x80,0xB6,0xEA,
  0xC9,0x22,0xD0,0x2B,0xA9,0x08,0x20,0xD2,0xFF,0xA9,0x80,0x20,0xD2,0xFF,0xA9,0x83,
  0x20,0xD2,0xFF,0xA9,0x80,0x20,0xD2,0xFF,0xA9,0x80,0x20,0xD2,0xFF,0xA9,0x83,0x20,
  0xD2,0xFF,0xA9,0x80,0x20,0xD2,0xFF,0xA9,0x0F,0x20,0xD2,0xFF,0x4C,0xB2,0xB6,0x20,
  0xD2,0xFF,0xA9,0x00,0x85,0xD4,0xA9,0x91,0x20,0xD2,0xFF,0xA9,0x92,0x20,0xD2,0xFF,
  0xC8,0xC0,0x28,0xF0,0x03,0x4C,0x38,0xB6,0xA9,0x0D,0x20,0xD2,0xFF,0xA0,0x00,0x18,
  0xA5,0x20,0x69,0x28,0x85,0x20,0xA5,0x21,0x69,0x00,0x85,0x21,0xE8,0xE0,0x19,0xD0,
  0xE2,0x20,0xCC,0xFF,0xA9,0x01,0x20,0xC3,0xFF,0x20,0xCC,0xFF,0x4C,0x57,0x82,0x20,
  0x31,0x80,0x4C,0xF5,0xBC,0x4C,0x61,0xBD,0x20,0x2F,0x82,0xE0,0x00,0xF0,0xF6,0xE0,
  0x80,0xF0,0xEC,0xE0,0x11,0x90,0x05,0xA9,0x0C,0x4C,0x4A,0x92,0xCA,0x86,0xAA,0xA9,
  0x00,0x85,0xAB,0x06,0xAA,0x26,0xAB,0x06,0xAA,0x26,0xAB,0x06,0xAA,0x26,0xAB,0x06,
  0xAA,0x26,0xAB,0x18,0xA5,0xAA,0x69,0x4B,0x85,0xAA,0xA5,0xAB,0x69,0xC6,0x85,0xAB,
  0x20,0x31,0x80,0x20,0x58,0x90,0x85,0xA9,0x86,0xA8,0xA0,0x00,0xB1,0xA8,0x91,0xAA,
  0xC8,0xC4,0x69,0xF0,0x04,0xC0,0x0F,0x90,0xF3,0xA9,0x00,0x91,0xAA,0x4C,0x57,0x82,
  0x20,0x4E,0x90,0x20,0xF0,0x88,0x8D,0x9A,0xC5,0x85,0x0A,0x8C,0x99,0xC5,0x84,0x09,
  0x20,0x29,0x82,0x8E,0xAA,0xC5,0x86,0xA4,0x20,0x9C,0x88,0xA9,0x00,0x85,0x65,0x85,
  0x64,0x85,0x62,0x85,0x59,0x85,0x57,0x85,0x58,0x85,0x5A,0xA5,0xF7,0x85,0x66,0xA9,
  0x0B,0x85,0xF7,0xA9,0x00,0x85,0x90,0x20,0x32,0x87,0xA5,0x90,0x85,0x6A,0xC5,0x66,
  0xD0,0x03,0x4C,0x57,0x82,0xAE,0x99,0xC5,0x86,0x5F,0xAD,0x9A,0xC5,0x85,0x60,0xAC,
  0xAA,0xC5,0x84,0x61,0xA5,0x60,0xA6,0x5F,0xD0,0x03,0x38,0xE9,0x01,0xCA,0xA4,0x61,
  0x20,0xFE,0xBA,0xD0,0x0B,0xA5,0x5F,0xD0,0x02,0xC6,0x60,0xC6,0x5F,0x4C,0xA4,0xB7,
  0xA5,0x5F,0x8D,0x99,0xC5,0xA5,0x60,0x8D,0x9A,0xC5,0x4C,0x3D,0xB9,0xA5,0x5B,0x85,
  0x57,0xA5,0x5C,0x85,0x58,0xA5,0x5D,0x85,0x59,0xA5,0x5E,0x85,0x5A,0xA9,0x01,0x85,
  0x5B,0x85,0x5C,0xAD,0x2E,0xC5,0xC9,0x0A,0xF0,0x13,0xA6,0x5F,0xA5,0x60,0xA4,0x61,
  0x88,0x20,0xFE,0xBA,0x85,0x5B,0xAD,0x2E,0xC5,0xC9,0x05,0xF0,0x0C,0xA6,0x5F,0xA5,
  0x60,0xA4,0x61,0xC8,0x20,0xFE,0xBA,0x85,0x5C,0xA6,0x5F,0xA5,0x60,0xE8,0xD0,0x03,
  0x18,0x69,0x01,0xA4,0x61,0x20,0xFE,0xBA,0x85,0x5D,0xA5,0x60,0xA6,0x5F,0xD0,0x03,
  0x38,0xE9,0x01,0xCA,0xA4,0x61,0x20,0xFE,0xBA,0x85,0x5E,0x18,0xA5,0x5B,0x65,0x5C,
  0x65,0x5D,0x65,0x5E,0xC9,0x04,0xF0,0x5B,0xA5,0x5B,0xC5,0x57,0xD0,0x46,0xA5,0x5C,
  0xC5,0x58,0xD0,0x32,0xA5,0x5D,0xC5,0x59,0xD0,0x29,0xA5,0x5E,0xC5,0x5A,0xD0,0x23,
  0xA9,0x01,0x85,0x62,0xA5,0x5B,0xD0,0x05,0xC6,0x61,0x4C,0x49,0xBA,0xA5,0x5C,0xF0,
  0x09,0xA5,0x5B,0x49,0x01,0x85,0x57,0x4C,0x84,0xB8,0xA9,0x00,0x85,0x62,0xE6,0x61,
  0x4C,0x49,0xBA,0x4C,0xA7,0xB9,0xA5,0x5C,0xD0,0x04,0xA5,0x58,0xD0,0xDF,0xA5,0x5B,
  0xD0,0x11,0xF0,0xCC,0xA5,0x5B,0xD0,0x07,0xA5,0x57,0xF0,0x03,0x4C,0x6F,0xBA,0xA5,
  0x5C,0xF0,0xCA,0xA5,0x5D,0xD0,0x35,0xA5,0x60,0xA6,0x5F,0xD0,0x03,0x38,0xE9,0x01,
  0xCA,0xA4,0x61,0x20,0x1F,0xBB,0xE6,0x5F,0xD0,0x02,0xE6,0x60,0xA5,0x60,0xA6,0x5F,
  0xE8,0xD0,0x03,0x18,0x69,0x01,0xA4,0x61,0x20,0x1F,0xBB,0xA5,0x61,0x8D,0x4D,0xCA,
  0xA5,0x65,0xC9,0x2D,0x90,0x03,0x20,0xB9,0xBA,0x4C,0xA7,0xB9,0xA5,0x5E,0xD0,0x28,
  0xA5,0x60,0xA6,0x5F,0xE8,0xD0,0x03,0x18,0x69,0x01,0xA4,0x61,0x20,0x1F,0xBB,0xA5,
  0x5F,0xD0,0x02,0xC6,0x60,0xC6,0x5F,0x38,0xA5,0x5F,0xE9,0x01,0xAA,0xA5,0x60,0xE9,
  0x00,0xA4,0x61,0x20,0x1F,0xBB,0xD0,0xC3,0xAD,0x4D,0xCA,0xC5,0x61,0xF0,0x1D,0xA5,
  0x61,0xCD,0x4D,0xCA,0xF0,0x37,0xB0,0x09,0xA9,0x0A,0x8D,0x2D,0xC5,0xE6,0x61,0xD0,
  0x2C,0xA9,0x05,0x8D,0x2D,0xC5,0xC6,0x61,0xF0,0x23,0xD0,0x21,0xA6,0x65,0xF0,0x67,
  0xCA,0x86,0x65,0xBD,0x4F,0xC7,0x85,0x60,0xBD,0x4E,0xC8,0x85,0x5F,0xBD,0x4D,0xC9,
  0x85,0x61,0xA6,0x5F,0xA5,0x60,0xA4,0x61,0x20,0xFE,0xBA,0xD0,0xDF,0xA5,0x61,0x8D,
  0x4D,0xCA,0xA5,0x60,0xA6,0x5F,0xA4,0x61,0x88,0x20,0xFE,0xBA,0x49,0x01,0x85,0x5B,
  0xA5,0x60,0xA6,0x5F,0xA4,0x61,0xC8,0x20,0xFE,0xBA,0x49,0x01,0x85,0x5C,0xA5,0x60,
  0xA6,0x5F,0xE8,0xD0,0x03,0x18,0x69,0x01,0xA4,0x61,0x20,0xFE,0xBA,0x49,0x01,0x85,
  0x5D,0xA5,0x60,0xA6,0x5F,0xD0,0x03,0x38,0xE9,0x01,0xCA,0xA4,0x61,0x20,0xFE,0xBA,
  0x49,0x01,0x85,0x5E,0x4C,0x49,0xBA,0xAD,0xAA,0xC5,0xC9,0xFF,0xD0,0x03,0x4C,0x57,
  0x82,0xAD,0x99,0xC5,0x85,0x5F,0xAD,0x9A,0xC5,0x85,0x60,0xAD,0xAA,0xC5,0x85,0x61,
  0xA9,0xFF,0x8D,0xAA,0xC5,0xD0,0x96,0xA5,0x5D,0xF0,0x1A,0xA5,0x59,0xD0,0x29,0xA5,
  0x62,0xD0,0x25,0xA5,0x60,0xA6,0x5F,0xE8,0xD0,0x03,0x18,0x69,0x01,0xA4,0x61,0x88,
  0x20,0x1F,0xBB,0xD0,0x13,0xA5,0x59,0xF0,0x0F,0xA5,0x60,0xA6,0x5F,0xE8,0xD0,0x03,
  0x18,0x69,0x01,0xA4,0x61,0x20,0x1F,0xBB,0xA5,0x62,0xF0,0x1A,0xA5,0x59,0xD0,0x30,
  0xA5,0x5D,0xF0,0x2C,0xA5,0x60,0xA6,0x5F,0xE8,0xD0,0x03,0x18,0x69,0x01,0xA4,0x61,
  0xC8,0x20,0x1F,0xBB,0xD0,0x1A,0xA5,0x5E,0xF0,0x1A,0xA5,0x5A,0xD0,0x29,0xA5,0x60,
  0xA6,0x5F,0xD0,0x03,0x38,0xE9,0x01,0xCA,0xA4,0x61,0x88,0x20,0x1F,0xBB,0xD0,0x17,
  0xA5,0x5E,0xD0,0x13,0xA5,0x5A,0xF0,0x0F,0xA5,0x60,0xA6,0x5F,0xD0,0x03,0x38,0xE9,
  0x01,0xCA,0xA4,0x61,0x20,0x1F,0xBB,0xA5,0x5E,0xC5,0x5A,0xF0,0x1C,0xA5,0x5E,0xF0,
  0x18,0xA5,0x5A,0xD0,0x14,0xA5,0x62,0xF0,0x10,0xA5,0x60,0xA6,0x5F,0xD0,0x03,0x38,
  0xE9,0x01,0xCA,0xA4,0x61,0xC8,0x20,0x1F,0xBB,0xA5,0x5F,0x85,0x09,0xA5,0x60,0x85,
  0x0A,0xA5,0x61,0x85,0xA4,0xAD,0x2D,0xC5,0x8D,0x2E,0xC5,0xA9,0x00,0x8D,0x2D,0xC5,
  0xA5,0x66,0x85,0xF7,0x20,0x32,0x87,0xA5,0x61,0x8D,0x4F,0xCA,0x4C,0xCD,0xB7,0xA5,
  0x5C,0xC5,0x58,0xD0,0x07,0xA9,0x00,0x85,0x64,0x4C,0x50,0xB8,0xA5,0x61,0xCD,0x4F,
  0xCA,0xD0,0x04,0xA5,0x64,0xF0,0x07,0xA9,0x00,0x85,0x64,0x4C,0x76,0xB8,0xA5,0x60,
  0x85,0x0A,0xA5,0x5F,0x85,0x09,0xA5,0x61,0x85,0xA4,0xA5,0x6A,0x85,0xF7,0x20,0x32,
  0x87,0xC6,0x61,0xA9,0x0A,0x85,0x64,0xE6,0x61,0xA5,0x60,0xA6,0x5F,0xA4,0x61,0x20,
  0xFE,0xBA,0xF0,0xF3,0xC6,0x61,0x4C,0x3D,0xB9,0xA9,0x00,0x8D,0x4C,0xCA,0xAE,0x4C,
  0xCA,0xBD,0x4D,0xC9,0xA8,0xBD,0x4E,0xC8,0x85,0xAA,0xBD,0x4F,0xC7,0xA6,0xAA,0x20,
  0xFE,0xBA,0xF0,0x1F,0xAE,0x4C,0xCA,0xBD,0x4F,0xC8,0x9D,0x4E,0xC8,0xBD,0x50,0xC7,
  0x9D,0x4F,0xC7,0xBD,0x4E,0xC9,0x9D,0x4D,0xC9,0xE8,0xE4,0x65,0xD0,0xE9,0xC6,0x65,
  0x4C,0xF6,0xBA,0xEE,0x4C,0xCA,0xAD,0x4C,0xCA,0xC5,0x65,0xD0,0xC1,0x60,0x86,0x09,
  0x85,0x0A,0x84,0xA4,0xA9,0x0B,0x85,0xF7,0xA9,0x00,0x85,0x90,0x20,0x32,0x87,0xA5,
  0x90,0xC5,0x6A,0xF0,0x05,0xA9,0x01,0x85,0x90,0x60,0xA9,0x00,0x85,0x90,0x60,0x86,
  0xAE,0xA6,0x65,0x9D,0x4F,0xC7,0x98,0x9D,0x4D,0xC9,0xA5,0xAE,0x9D,0x4E,0xC8,0xE6,
  0x65,0x60,0x20,0x4E,0x90,0xAD,0x2F,0xC5,0x48,0xAD,0x30,0xC5,0x48,0xA9,0x01,0x8D,
  0x30,0xC5,0xA9,0x3F,0x8D,0x2F,0xC5,0x20,0xCC,0xFF,0xA9,0x01,0x20,0xC3,0xFF,0x20,
  0xCC,0xFF,0xAD,0xAE,0xC5,0x48,0xA9,0x00,0x8D,0xAE,0xC5,0xA2,0x04,0xA9,0x01,0xA0,
  0xFF,0x20,0xBA,0xFF,0x20,0xC0,0xFF,0xA2,0x01,0x20,0xC9,0xFF,0xA9,0x0D,0x20,0xD2,
  0xFF,0xA9,0x08,0x20,0xD2,0xFF,0xA9,0x00,0x85,0x5F,0x85,0x61,0x85,0x60,0xA9,0x00,
  0x8D,0x4C,0xCA,0xA9,0x00,0x8D,0xE0,0xC5,0xA9,0x00,0x8D,0xE4,0xC5,0x18,0xA5,0x5F,
  0x6D,0x4C,0xCA,0x85,0x09,0xA5,0x60,0x69,0x00,0x85,0x0A,0x18,0xA5,0x61,0x6D,0xE4,
  0xC5,0x85,0xA4,0xA9,0x0B,0x85,0xF7,0xA9,0x00,0x85,0x90,0x20,0x32,0x87,0xA5,0x90,
  0xF0,0x10,0xC9,0x08,0xF0,0x0C,0xAD,0xE0,0xC5,0xAC,0xE4,0xC5,0x19,0xC0,0x9D,0x8D,
  0xE0,0xC5,0xEE,0xE4,0xC5,0xAD,0xE4,0xC5,0xC9,0x07,0xD0,0xC1,0xAD,0xE0,0xC5,0x09,
  0x80,0x20,0xD2,0xFF,0xEE,0x4C,0xCA,0xAD,0x4C,0xCA,0xC9,0x06,0xF0,0x03,0x4C,0x83,
  0xBB,0x18,0xA5,0x5F,0x69,0x06,0x85,0x5F,0xA5,0x60,0x69,0x00,0x85,0x60,0xA5,0x5F,
  0xC9,0x44,0xD0,0x06,0xA5,0x60,0xC9,0x01,0xF0,0x03,0x4C,0x7E,0xBB,0xA9,0x0D,0x20,
  0xD2,0xFF,0x18,0xA5,0x61,0x69,0x07,0x85,0x61,0xA9,0x00,0x85,0x5F,0x85,0x60,0xA5,
  0x61,0xC9,0xD2,0x90,0xE5,0xA9,0x0D,0x20,0xD2,0xFF,0xA9,0x0F,0x20,0xD2,0xFF,0x20,
  0xCC,0xFF,0xA9,0x01,0x20,0xC3,0xFF,0x20,0xCC,0xFF,0x68,0x8D,0xAE,0xC5,0x68,0x8D,
  0x30,0xC5,0x68,0x8D,0x2F,0xC5,0x4C,0x57,0x82,0x20,0x7B,0x88,0xAD,0xA6,0xC5,0x85,
  0x61,0xAD,0xA8,0xC5,0x85,0x5F,0xAD,0xA9,0xC5,0x85,0x60,0x18,0xAD,0x99,0xC5,0x65,
  0x5F,0x8D,0xA8,0xC5,0xAD,0x9A,0xC5,0x65,0x60,0x8D,0xA9,0xC5,0xAD,0xAA,0xC5,0x8D,
  0xA6,0xC5,0x20,0xB5,0xBC,0x18,0xAD,0xA6,0xC5,0x8D,0xAA,0xC5,0x65,0x61,0x8D,0xA6,
  0xC5,0xAD,0xA8,0xC5,0x8D,0x99,0xC5,0xAD,0xA9,0xC5,0x8D,0x9A,0xC5,0x20,0xB5,0xBC,
  0x38,0xAD,0x99,0xC5,0xE5,0x5F,0x8D,0x99,0xC5,0xAD,0x9A,0xC5,0xE5,0x60,0x8D,0x9A,
  0xC5,0xAD,0xA6,0xC5,0x8D,0xAA,0xC5,0x20,0xB5,0xBC,0x38,0xAD,0xAA,0xC5,0xE5,0x61,
  0x8D,0xAA,0xC5,0xAD,0x99,0xC5,0x8D,0xA8,0xC5,0xAD,0x9A,0xC5,0x8D,0xA9,0xC5,0x20,
  0xB5,0xBC,0x4C,0x57,0x82,0xAD,0x99,0xC5,0x85,0x5B,0xAD,0x9A,0xC5,0x85,0x5C,0xAD,
  0xA8,0xC5,0x85,0x5D,0xAD,0xA9,0xC5,0x85,0x5E,0xAD,0xA6,0xC5,0x85,0x59,0xAD,0xAA,
  0xC5,0x85,0x57,0x20,0xF6,0x84,0xA5,0x5B,0x8D,0x99,0xC5,0xA5,0x5C,0x8D,0x9A,0xC5,
  0xA5,0x5D,0x8D,0xA8,0xC5,0xA5,0x5E,0x8D,0xA9,0xC5,0xA5,0x59,0x8D,0xA6,0xC5,0xA5,
  0x57,0x8D,0xAA,0xC5,0x60,0x4C,0xA2,0xA0,0xA9,0xCC,0x8D,0x88,0x02,0xA9,0x93,0x20,
  0xD2,0xFF,0xA9,0x00,0x85,0x20,0xA9,0xD0,0x85,0x21,0x78,0xA5,0x01,0x29,0xFB,0x85,
  0x01,0xA9,0x00,0x85,0xA8,0xA9,0xE0,0x85,0xA9,0xA0,0x00,0xA2,0x00,0xB1,0x20,0x91,
  0xA8,0xE6,0x20,0xD0,0x03,0xE6,0x21,0xE8,0x20,0x61,0x89,0xE0,0x10,0xD0,0xEE,0xA5,
  0x01,0x09,0x04,0x85,0x01,0xA9,0x94,0x8D,0x00,0xDD,0xA9,0x39,0x8D,0x18,0xD0,0x4C,
  0x03,0x86,0x20,0x2F,0x82,0x8E,0x17,0xC5,0xBD,0x1E,0xD0,0x8D,0x16,0xC5,0x4C,0x57,
  0x82,0x4B,0x45,0x59,0x00,0x43,0x48,0x52,0x24,0x28,0x31,0x33,0x29,0x00,0x20,0x4E,
  0x90,0xA2,0x00,0x86,0xA7,0xA9,0x01,0x85,0xA6,0xA9,0xC6,0x85,0x21,0xA9,0x4B,0x85,
  0x20,0xA0,0x00,0xB9,0x51,0xBD,0xF0,0x06,0x20,0xD2,0xFF,0xC8,0xD0,0xF5,0xA5,0xA6,
  0x85,0x63,0xA5,0xA7,0x85,0x62,0xA2,0x90,0x38,0x20,0x88,0x80,0x20,0xB5,0x80,0x20,
  0xB1,0xA1,0xA9,0x2C,0x20,0xD2,0xFF,0xA9,0x22,0x20,0xD2,0xFF,0xA9,0x01,0x8D,0xAA,
  0xC5,0xA2,0x00,0xA0,0x00,0xB1,0x20,0xF0,0x22,0xC9,0x0D,0xF0,0x45,0xAD,0xAA,0xC5,
  0xD0,0x0F,0xA9,0x2C,0x20,0xD2,0xFF,0xA9,0x22,0x20,0xD2,0xFF,0xA9,0x01,0x8D,0xAA,
  0xC5,0xB1,0x20,0x20,0xD2,0xFF,0xE8,0xC8,0x4C,0xA5,0xBD,0xE6,0xA6,0x18,0xA5,0x20,
  0x69,0x10,0x85,0x20,0xA5,0x21,0x69,0x00,0x85,0x21,0xAD,0xAA,0xC5,0xF0,0x05,0xA9,
  0x22,0x20,0xD2,0xFF,0xA9,0x0D,0x20,0xD2,0xFF,0xA5,0xA6,0xC9,0x11,0xD0,0x82,0x4C,
  0x57,0x82,0xAD,0xAA,0xC5,0xF0,0x0F,0xA9,0x22,0x20,0xD2,0xFF,0xA9,0x2B,0x20,0xD2,
  0xFF,0xA9,0x00,0x8D,0xAA,0xC5,0xA2,0x00,0xBD,0x55,0xBD,0xF0,0x06,0x20,0xD2,0xFF,
  0xE8,0xD0,0xF5,0x4C,0xC6,0xBD,0x08,0x8C,0xFA,0x03,0x8E,0xFB,0x03,0xAC,0x8D,0x02,
  0xC0,0x01,0xD0,0x0D,0xA2,0x00,0xC8,0xD0,0xFD,0xE8,0xEC,0xF6,0xC5,0xD0,0xF7,0xF0,
  0x0B,0xAC,0x8D,0x02,0xC0,0x04,0xF0,0x04,0xC0,0x00,0xD0,0xF5,0xAC,0x8E,0xC5,0xC0,
  0x0A,0xD0,0x43,0xAC,0x8F,0xC5,0xC4,0xD6,0xB0,0x3C,0xA0,0x00,0x48,0xB1,0xD1,0x99,
  0x44,0x02,0xC9,0x20,0xF0,0x04,0xC8,0x4C,0x4D,0xBE,0x20,0xE4,0xFF,0xC9,0x0D,0xD0,
  0xF9,0xA5,0xD4,0x48,0xA9,0x00,0x85,0xD4,0xA9,0x93,0x20,0xD2,0xFF,0xA0,0x00,0xA9,
  0x20,0x20,0xD2,0xFF,0xB9,0x44,0x02,0xC9,0x20,0xF0,0x07,0x99,0x00,0x04,0xC8,0x4C,
  0x6F,0xBE,0x68,0x85,0xD4,0x68,0xA2,0xA5,0x86,0x21,0xA2,0x21,0x86,0x20,0xC9,0x64,
  0xF0,0x09,0xAC,0xFA,0x03,0xAE,0xFB,0x03,0x4C,0x77,0x82,0xA6,0xD4,0xE0,0x00,0xD0,
  0xF1,0xA0,0x00,0x84,0x22,0xAC,0xFA,0x03,0xC8,0xB1,0x5F,0xAA,0xCA,0x8A,0xA0,0x00,
  0x8D,0xD3,0xC5,0x48,0xAD,0xDA,0xC5,0xC9,0x0A,0xD0,0x05,0xA9,0x12,0x20,0xD2,0xFF,
  0x68,0xC9,0x00,0xF0,0x1D,0xB1,0x20,0xC9,0x40,0xF0,0x08,0xE6,0x20,0xD0,0xF6,0xE6,
  0x21,0xD0,0xF2,0xE6,0x22,0xA5,0x22,0xCD,0xD3,0xC5,0xD0,0xEF,0xE6,0x20,0xD0,0x02,
  0xE6,0x21,0xB1,0x20,0xC9,0x40,0xF0,0x06,0x20,0xD2,0xFF,0xC8,0xD0,0xF4,0xA9,0x92,
  0x20,0xD2,0xFF,0xAC,0xFA,0x03,0xC8,0xAE,0xFB,0x03,0xA9,0x02,0x4C,0x80,0x82,0x20,
  0xD2,0xFF,0xA5,0x66,0x10,0x05,0xA9,0x2D,0x20,0xD2,0xFF,0x60,0xAD,0xE6,0xC5,0x85,
  0x7A,0xAD,0xE7,0xC5,0x85,0x7B,0xA9,0x00,0x8D,0xE8,0xC5,0x60,0xAD,0xEF,0xC5,0x85,
  0x7A,0xA9,0x00,0x8D,0x4E,0xC7,0x60,0xA9,0x01,0x8D,0x4E,0xC7,0xAD,0xEA,0xC5,0x85,
  0x90,0x68,0x8D,0xEE,0xC5,0x68,0x8D,0xEF,0xC5,0x68,0x68,0x4C,0x57,0x82,0xA9,0x00,
  0x8D,0x4E,0xC7,0x4C,0x4E,0x90,0x20,0x71,0xBF,0xC8,0xB1,0x20,0x30,0x02,0x88,0x60,
  0x88,0x20,0x06,0xBF,0x20,0xBF,0xA1,0xB1,0x20,0x49,0xFF,0x85,0x62,0x20,0xBF,0xA1,
  0xB1,0x20,0x49,0xFF,0x85,0x63,0xE6,0x63,0xD0,0x02,0xE6,0x62,0x68,0x68,0x4C,0x16,
  0xA1,0xA9,0x3D,0x4C,0xD2,0xFF,0xA2,0x00,0x86,0x33,0x86,0x37,0xA9,0x6D,0x8D,0x06,
  0x03,0xA9,0x82,0x8D,0x07,0x03,0xA9,0x80,0x85,0x38,0x85,0x34,0xA9,0x09,0x8D,0x08,
  0x03,0x60,0xAA,0xAA,0xAA,0xAD,0x1B,0xCB,0xF0,0x61,0xA5,0xAA,0x48,0xA5,0xAB,0x48,
  0x8A,0x48,0x98,0x48,0xA5,0x09,0x8D,0x92,0xBF,0xA5,0x0A,0x8D,0x93,0xBF,0xAD,0xAE,
  0xC5,0xF0,0x06,0x0E,0x92,0xBF,0x2E,0x93,0xBF,0xAD,0x93,0xBF,0x4A,0x6E,0x92,0xBF,
  0x4A,0x6E,0x92,0xBF,0x4A,0x6E,0x92,0xBF,0xA5,0xA4,0x4A,0x4A,0x4A,0xA8,0x18,0xB9,
  0x28,0x89,0x6D,0x92,0xBF,0x85,0xAA,0xB9,0x41,0x89,0x69,0xC0,0x85,0xAB,0xA0,0x00,
  0xAD,0x19,0xCB,0x91,0xAA,0x18,0xA5,0xAB,0x69,0x18,0x85,0xAB,0xAD,0x1A,0xCB,0x91,
  0xAA,0x68,0xA8,0x68,0xAA,0x68,0x85,0xAB,0x68,0x85,0xAA,0xA5,0xF7,0xC9,0x02,0x60
};
